<?php
session_start();
header('Content-Type: text/html; charset=utf-8');
include '../includes/db.php';
include '../includes/auth.php';
checkAuth();

// Establecer la codificación de caracteres para la conexión de la base de datos
$conn->set_charset("utf8");

// Obtener el ID del usuario que ha iniciado sesión
$usuario_id = $_SESSION['usuario_id'];

// Obtener el semestre seleccionado por el usuario
$semestre = isset($_POST['semestre']) ? intval($_POST['semestre']) : 1; // Default a semestre 1

// --- CORREGIDO: Consulta SQL para incluir notas E indicadores ---
try {
    $sql = "
    SELECT * FROM (
        SELECT asignatura.nombre AS asignatura, notas.nota, NULL AS indicador, notas.numero_nota, notas.semestre 
        FROM notas 
        JOIN asignatura ON notas.asignatura_id = asignatura.id 
        WHERE notas.usuario_id = ? AND notas.semestre = ?
        UNION
        SELECT asignatura.nombre AS asignatura, NULL AS nota, indicadores.nombre AS indicador, notas_indicadores.numero_nota, notas_indicadores.semestre 
        FROM notas_indicadores 
        JOIN asignatura ON notas_indicadores.asignatura_id = asignatura.id 
        JOIN indicadores ON notas_indicadores.indicador_id = indicadores.id 
        WHERE notas_indicadores.usuario_id = ? AND notas_indicadores.semestre = ?
    ) AS combined_results
    ORDER BY asignatura, numero_nota";
    
    $stmt = $conn->prepare($sql);
    // Necesitamos 4 parámetros para la consulta UNION
    $stmt->bind_param("iiii", $usuario_id, $semestre, $usuario_id, $semestre);
    $stmt->execute();
    $notas = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
} catch (Exception $e) {
    $error_message = $e->getMessage();
}

// --- CORREGIDO: Organizar las notas (distinguiendo numéricas de conceptuales) ---
$notas_por_asignatura = [];
$es_indicador = []; // Array para rastrear qué asignaturas son conceptuales
foreach ($notas as $nota) {
    $asignatura = $nota['asignatura'];
    if ($nota['indicador'] !== null) {
        // Es un indicador (MB, B, S, I)
        $notas_por_asignatura[$asignatura][$nota['numero_nota']] = $nota['indicador'];
        $es_indicador[$asignatura] = true;
    } else {
        // Es una nota numérica
        $notas_por_asignatura[$asignatura][$nota['numero_nota']] = $nota['nota'];
        $es_indicador[$asignatura] = false;
    }
}

// --- (Función de promedio numérico) ---
function calcular_promedio($notas) {
    $notas_numericas = array_filter($notas, 'is_numeric');
    if (empty($notas_numericas)) return 0;
    return number_format(array_sum($notas_numericas) / count($notas_numericas), 1);
}

// --- AÑADIDO: Función de promedio de indicadores (copiada de reports.php) ---
function promedio_indicadores($indicadores) {
    $puntajes = [];
    foreach ($indicadores as $indicador) {
        switch ($indicador) {
            case 'MB': $puntajes[] = 7.0; break;
            case 'B':  $puntajes[] = 5.5; break;
            case 'S':  $puntajes[] = 4.5; break;
            case 'I':  $puntajes[] = 2.5; break;
        }
    }
    if (empty($puntajes)) return 'I';
    $promedio_num = array_sum($puntajes) / count($puntajes);
    
    if ($promedio_num >= 6.0) return 'MB';
    elseif ($promedio_num >= 5.0) return 'B';
    elseif ($promedio_num >= 4.0) return 'S';
    else return 'I';
}

$promedios_finales = []; // Array para el promedio final (solo numéricas)
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../images/logo.png">
    <title>Mis Notas</title>
    
    <link rel="stylesheet" href="../css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    
    <style>
        .notas-table {
            width: 100%;
            min-width: 800px;
            border-collapse: collapse;
            margin-top: 1.5rem;
        }
        .notas-table th, .notas-table td {
            border: 1px solid var(--grey-medium);
            padding: 0.75rem;
            text-align: center;
            vertical-align: middle;
        }
        .notas-table thead th {
            background-color: var(--grey-light);
            font-size: 0.9rem;
            color: var(--dark-color);
        }
        .notas-table tbody th {
            text-align: left;
            background-color: #fdfdfd;
            font-weight: 700;
        }
        .notas-table tfoot td {
            text-align: right;
            font-weight: 700;
            font-size: 1.1rem;
            color: var(--dark-color);
        }
        .notas-table tfoot .promedio-final {
            text-align: center;
            background-color: var(--secondary-color);
            color: white;
            font-size: 1.2rem;
        }
        .nota-roja { color: var(--danger-color); font-weight: 700; }
        .nota-azul { color: var(--primary-color); font-weight: 700; }
        .nota-conceptual { color: var(--dark-color); font-weight: 700; font-style: italic; }
    </style>
</head>
<body>
    
<header>
    <div class="header-content">
        <img src="../images/INSIGNIA SF.png" alt="Escuela Porongo Insignia">
        <h1>Mis Notas</h1>
        <nav>
            <ul>
                <li><a href="../index.php"><i class="fas fa-home"></i> Inicio</a></li>
            </ul>
        </nav>
    </div>
</header>
    
<main class="container">
    <div class="card table-container">
        
        <div class="table-header" style="padding-bottom: 1rem; border-bottom: 2px solid var(--grey-light);">
            <h2><i class="fas fa-book-open"></i> Calificaciones</h2>
            <form method="POST" class="search-container" style="gap: 0.5rem;">
                <div class="form-group" style="margin-bottom: 0;">
                    <label for="semestre" style="margin-bottom: 0; margin-right: 0.5rem;">Seleccione el semestre:</label>
                    <select name="semestre" id="semestre" style="padding: 0.5rem;">
                        <option value="1" <?php echo ($semestre == 1) ? 'selected' : ''; ?>>Primer Semestre</option>
                        <option value="2" <?php echo ($semestre == 2) ? 'selected' : ''; ?>>Segundo Semestre</option>
                    </select>
                </div>
                <button type="submit" class="btn btn-primary"><i class="fas fa-eye"></i> Ver Notas</button>
            </form>
        </div>

        <?php if (isset($error_message)): ?>
            <p>Error: <?php echo htmlspecialchars($error_message, ENT_QUOTES, 'UTF-8'); ?></p>
        
        <?php elseif (!empty($notas)): ?>
            <div class="table-responsive">
                <table class="notas-table">
                    <thead>
                        <tr>
                            <th>Asignatura</th>
                            <?php for ($i = 1; $i <= 10; $i++): ?>
                                <th>N<?php echo $i; ?></th>
                            <?php endfor; ?>
                            <th>Promedio</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        // --- CORREGIDO: Bucle principal ---
                        foreach ($notas_por_asignatura as $asignatura => $notas_list):
                            $es_asig_indicador = $es_indicador[$asignatura] ?? false;
                            $promedio_asignatura_display = '0';
                            $clase_promedio = 'nota-azul';
                            
                            if ($es_asig_indicador) {
                                // Es Orientación o Religión
                                $promedio_asignatura_display = promedio_indicadores($notas_list);
                                $clase_promedio = 'nota-conceptual';
                            } else {
                                // Es una asignatura numérica
                                $promedio_num = calcular_promedio($notas_list);
                                $promedio_asignatura_display = $promedio_num;
                                $clase_promedio = $promedio_num < 4.0 ? 'nota-roja' : 'nota-azul';
                                // AÑADIDO: Solo agregar al promedio final si es numérica
                                if ($promedio_num > 0) {
                                    $promedios_finales[] = $promedio_num;
                                }
                            }
                        ?>
                            <tr>
                                <th><?php echo htmlspecialchars($asignatura, ENT_QUOTES, 'UTF-8'); ?></th>
                                
                                <?php for ($i = 1; $i <= 10; $i++): 
                                    $nota = $notas_list[$i] ?? null;
                                    $clase_nota = '';
                                    if ($nota !== null) {
                                        if ($es_asig_indicador) {
                                            $clase_nota = 'nota-conceptual';
                                        } else {
                                            $clase_nota = floatval($nota) < 4.0 ? 'nota-roja' : 'nota-azul';
                                        }
                                    }
                                ?>
                                    <td class="<?php echo $clase_nota; ?>">
                                        <?php echo $nota !== null ? htmlspecialchars($nota, ENT_QUOTES, 'UTF-8') : '—'; ?>
                                    </td>
                                <?php endfor; ?>
                                
                                <td class="<?php echo $clase_promedio; ?>">
                                    <?php echo htmlspecialchars($promedio_asignatura_display, ENT_QUOTES, 'UTF-8'); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot>
                        <tr>
                            <td colspan="11" style="text-align: right;">Promedio Final (Solo asignaturas numéricas)</td>
                            <td class="promedio-final">
                                <?php echo htmlspecialchars(calcular_promedio($promedios_finales), ENT_QUOTES, 'UTF-8'); ?>
                            </td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        <?php else: ?>
            <div style="text-align: center; padding: 2rem; color: var(--grey-dark);">
                No se encontraron notas ni indicadores para el semestre seleccionado.
            </div>
        <?php endif; ?>
    </div>
</main>
</body>
</html>