<?php
session_start();
header('Content-Type: text/html; charset=utf-8');
include '../includes/db.php';
include '../includes/auth.php';
checkAuth();
$conn->set_charset("utf8");

// --- 1. MANEJO DE MENSAJES ---
$message = ''; $message_type = '';
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message'], $_SESSION['message_type']);
}

// --- 2. LÓGICA: CREAR USUARIO DESDE MATRÍCULA ---
if (isset($_GET['add_user_id'])) {
    $matricula_id = intval($_GET['add_user_id']);
    // Obtenemos datos incluyendo el curso
    $sql = "SELECT rut, nombre, apellido_paterno, apellido_materno, curso_id, email_contacto FROM matricula WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $matricula_id);
    $stmt->execute();
    $matricula = $stmt->get_result()->fetch_assoc();

    if ($matricula) {
        // Verificar duplicado
        $check = $conn->prepare("SELECT id FROM usuario WHERE rut = ?");
        $check->bind_param("s", $matricula['rut']);
        $check->execute();
        
        if ($check->get_result()->num_rows > 0) {
            $_SESSION['message'] = "⚠️ El usuario con RUT " . $matricula['rut'] . " ya existe.";
            $_SESSION['message_type'] = 'warning';
        } else {
            $rol_id = 3; // 3 = Estudiante
            // Contraseña = RUT sin puntos ni guion (o el mismo RUT completo si prefieres)
            $password = password_hash($matricula['rut'], PASSWORD_BCRYPT); 
            $email = $matricula['email_contacto'] ?? ''; // Opcional si tienes ese campo

            $sql_ins = "INSERT INTO usuario (rut, nombre, apellido_paterno, apellido_materno, email, password, rol_id, curso_id, activo) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 1)";
            $stmt_ins = $conn->prepare($sql_ins);
            $stmt_ins->bind_param("ssssssii", $matricula['rut'], $matricula['nombre'], $matricula['apellido_paterno'], $matricula['apellido_materno'], $email, $password, $rol_id, $matricula['curso_id']);

            if ($stmt_ins->execute()) {
                $_SESSION['message'] = "✅ Usuario de sistema creado para el estudiante.";
                $_SESSION['message_type'] = 'exito';
            } else {
                $_SESSION['message'] = "Error al crear usuario: " . $stmt_ins->error;
                $_SESSION['message_type'] = 'error';
            }
        }
    }
    header("Location: matricula_crud.php");
    exit();
}

// --- 3. LÓGICA: ELIMINAR MATRÍCULA ---
if (isset($_GET['delete_id'])) {
    $id = intval($_GET['delete_id']);
    $conn->begin_transaction();
    try {
        // Borrado en cascada manual (por seguridad)
        $conn->query("DELETE FROM previsiones_salud WHERE matricula_id = $id");
        $conn->query("DELETE FROM informacion_adicional WHERE matricula_id = $id");
        $conn->query("DELETE FROM familiares WHERE matricula_id = $id");
        $conn->query("DELETE FROM apoderados WHERE matricula_id = $id");
        $conn->query("DELETE FROM alergias_alimentarias WHERE matricula_id = $id");
        $conn->query("DELETE FROM matricula WHERE id = $id");
        
        $conn->commit();
        $_SESSION['message'] = "🗑️ Matrícula eliminada correctamente.";
        $_SESSION['message_type'] = 'exito';
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['message'] = "Error: " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    }
    header("Location: crud_matriculas.php");
    exit();
}

// --- 4. CONFIGURACIÓN DE PAGINACIÓN Y BÚSQUEDA ---
$limite = 15;
$pagina = isset($_GET['pagina']) ? max(1, intval($_GET['pagina'])) : 1;
$offset = ($pagina - 1) * $limite;

$search = $_GET['search_rut'] ?? '';
$whereSQL = "WHERE 1=1";
if (!empty($search)) {
    $safe_search = $conn->real_escape_string($search);
    $whereSQL .= " AND (matricula.rut LIKE '%$safe_search%' OR matricula.apellido_paterno LIKE '%$safe_search%')";
}

// Consulta Principal (Incluyendo Estado y Fecha)
$sql = "
    SELECT m.id, m.rut, m.nombre, m.apellido_paterno, m.apellido_materno, m.estado, m.fecha_registro,
           c.nombre AS curso_nombre
    FROM matricula m
    LEFT JOIN curso c ON m.curso_id = c.id
    $whereSQL
    ORDER BY m.id DESC
    LIMIT $offset, $limite
";
$matriculas = $conn->query($sql);

// Total para paginación
$sql_count = "SELECT COUNT(*) as total FROM matricula $whereSQL";
$total_registros = $conn->query($sql_count)->fetch_assoc()['total'];
$total_paginas = ceil($total_registros / $limite);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../images/logo.png">
    <title>Gestión de Matrículas</title>
    
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="stylesheet" href="../css/admin.css">

    <style>
        /* Estilos Modernos */
        :root { --primary: #1e3a8a; --secondary: #3b82f6; --success: #10b981; --warning: #f59e0b; --danger: #ef4444; --light: #f8fafc; }
        body { font-family: 'Roboto', sans-serif; background-color: #e2e8f0; color: #334155; }
        .container { max-width: 1200px; margin: 30px auto; padding: 0 20px; }
        
        .card { background: white; border-radius: 12px; box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1); overflow: hidden; }
        
        /* Header de la Tabla */
        .table-header { padding: 20px; background: white; display: flex; justify-content: space-between; align-items: center; border-bottom: 1px solid #e2e8f0; flex-wrap: wrap; gap: 15px; }
        .table-header h2 { margin: 0; font-size: 1.5rem; color: var(--primary); display: flex; align-items: center; gap: 10px; }
        
        /* Buscador */
        .search-box { display: flex; gap: 0; background: #f1f5f9; border-radius: 8px; padding: 2px; border: 1px solid #cbd5e1; }
        .search-box input { border: none; background: transparent; padding: 8px 12px; outline: none; font-size: 0.95rem; width: 250px; }
        .search-box button { background: var(--secondary); color: white; border: none; padding: 8px 15px; border-radius: 6px; cursor: pointer; transition: 0.2s; }
        .search-box button:hover { background: var(--primary); }

        /* Tabla */
        .table-responsive { overflow-x: auto; }
        table { width: 100%; border-collapse: collapse; }
        thead { background-color: #f8fafc; }
        th { text-align: left; padding: 15px; font-weight: 600; color: #64748b; font-size: 0.85rem; text-transform: uppercase; border-bottom: 2px solid #e2e8f0; }
        td { padding: 15px; border-bottom: 1px solid #e2e8f0; font-size: 0.95rem; vertical-align: middle; }
        tr:hover { background-color: #f8fafc; }

        /* Badges de Estado */
        .badge { padding: 4px 10px; border-radius: 20px; font-size: 0.75rem; font-weight: 600; display: inline-block; }
        .badge-success { background-color: #dcfce7; color: #166534; } /* Verde */
        .badge-warning { background-color: #fef3c7; color: #92400e; } /* Amarillo */
        .badge-secondary { background-color: #e2e8f0; color: #475569; } /* Gris */

        /* Botones de Acción */
        .action-btn { display: inline-flex; align-items: center; justify-content: center; width: 32px; height: 32px; border-radius: 6px; color: white; margin-right: 5px; transition: 0.2s; text-decoration: none; font-size: 0.9rem; }
        .btn-view { background-color: var(--secondary); } .btn-view:hover { background-color: var(--primary); }
        .btn-edit { background-color: var(--warning); } .btn-edit:hover { background-color: #d97706; }
        .btn-del { background-color: var(--danger); } .btn-del:hover { background-color: #dc2626; }
        .btn-user { background-color: var(--success); } .btn-user:hover { background-color: #059669; }

        /* Paginación */
        .pagination { display: flex; justify-content: center; padding: 20px; gap: 5px; }
        .page-link { padding: 8px 14px; border: 1px solid #e2e8f0; border-radius: 6px; text-decoration: none; color: #64748b; background: white; transition: 0.2s; }
        .page-link.active { background-color: var(--primary); color: white; border-color: var(--primary); }
        .page-link:hover:not(.active) { background-color: #f1f5f9; }

        /* Mensajes */
        .mensaje { position: fixed; top: 90px; right: 20px; padding: 15px 25px; border-radius: 8px; color: white; font-weight: 500; box-shadow: 0 4px 10px rgba(0,0,0,0.1); z-index: 3000; animation: slideIn 0.3s; }
        .mensaje.exito { background: var(--success); }
        .mensaje.error { background: var(--danger); }
        .mensaje.warning { background: var(--warning); color: #fff; }
        @keyframes slideIn { from { transform: translateX(100%); } to { transform: translateX(0); } }
    </style>
</head>
<body>

<header>
    <div class="header-content">
        <img src="../images/INSIGNIA SF.png" alt="Logo">
        <h1>Administración</h1>
        <nav>
            <ul>
                <li><a href="../index.php"><i class="fas fa-home"></i> Inicio</a></li>
                <li><a href="formulario_matricula.php" class="btn btn-primary" style="background: white; color: var(--primary); padding: 5px 15px; border-radius: 20px; font-weight: bold;"><i class="fas fa-plus"></i> Nueva Matrícula</a></li>
            </ul>
        </nav>
    </div>
</header>

<main class="container">
    <?php if ($message): ?>
        <div class="mensaje <?= $message_type ?> show"><?= htmlspecialchars($message) ?></div>
        <script>setTimeout(() => document.querySelector('.mensaje').remove(), 4000);</script>
    <?php endif; ?>

    <div class="card">
        <div class="table-header">
            <h2><i class="fas fa-list-alt"></i> Listado de Matrículas</h2>
            
            <form method="GET" class="search-box">
                <input type="text" name="search_rut" placeholder="Buscar por RUT o Apellido..." value="<?= htmlspecialchars($search) ?>">
                <button type="submit"><i class="fas fa-search"></i></button>
            </form>
        </div>

        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>Folio</th>
                        <th>Fecha</th>
                        <th>Estado</th>
                        <th>RUT Alumno</th>
                        <th>Nombre Completo</th>
                        <th>Curso</th>
                        <th style="text-align: center;">Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($matriculas && $matriculas->num_rows > 0): ?>
                        <?php while ($row = $matriculas->fetch_assoc()): ?>
                            <?php 
                                // Color del estado
                                $estado = $row['estado'] ?? 'Completada';
                                $badgeClass = ($estado === 'Completada') ? 'badge-success' : 'badge-warning';
                                
                                // Nombre completo
                                $nombreComp = $row['nombre'] . ' ' . $row['apellido_paterno'] . ' ' . $row['apellido_materno'];
                            ?>
                            <tr>
                                <td><strong>#<?= str_pad($row['id'], 4, "0", STR_PAD_LEFT) ?></strong></td>
                                <td><?= date('d/m/Y', strtotime($row['fecha_registro'] ?? date('Y-m-d'))) ?></td>
                                <td><span class="badge <?= $badgeClass ?>"><?= $estado ?></span></td>
                                <td><?= htmlspecialchars($row['rut']) ?></td>
                                <td style="text-transform: uppercase; font-weight: 500;"><?= htmlspecialchars($nombreComp) ?></td>
                                <td><?= htmlspecialchars($row['curso_nombre'] ?? 'Sin Asignar') ?></td>
                                <td style="text-align: center;">
                                    
                                    <a href="ver_matricula.php?id=<?= $row['id'] ?>" class="action-btn btn-view" title="Ver Ficha">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    
                                    <a href="editar_matricula.php?id=<?= $row['id'] ?>" class="action-btn btn-edit" title="Editar Datos">
                                        <i class="fas fa-pencil-alt"></i>
                                    </a>
                                    
                                    <a href="?add_user_id=<?= $row['id'] ?>" class="action-btn btn-user" title="Crear Usuario de Sistema" onclick="return confirm('¿Crear usuario de acceso para <?= $row['nombre'] ?>? La contraseña será el RUT.')">
                                        <i class="fas fa-user-plus"></i>
                                    </a>

                                    <a href="?delete_id=<?= $row['id'] ?>" class="action-btn btn-del" title="Eliminar Matrícula" onclick="return confirm('⚠️ ¿ESTÁS SEGURO? \nSe borrarán TODOS los datos, familiares y apoderados asociados a esta matrícula.')">
                                        <i class="fas fa-trash-alt"></i>
                                    </a>

                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="7" style="text-align: center; padding: 30px; color: #64748b;">
                                <i class="fas fa-search" style="font-size: 2rem; display: block; margin-bottom: 10px; opacity: 0.5;"></i>
                                No se encontraron matrículas registradas.
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <?php if ($total_paginas > 1): ?>
        <div class="pagination">
            <?php 
            // Rango de páginas para no mostrar demasiadas
            $range = 2;
            $start = max(1, $pagina - $range);
            $end = min($total_paginas, $pagina + $range);
            
            if($pagina > 1) echo '<a href="?pagina='.($pagina-1).'&search_rut='.$search.'" class="page-link"><i class="fas fa-chevron-left"></i></a>';
            
            for ($i = $start; $i <= $end; $i++) {
                $active = ($i == $pagina) ? 'active' : '';
                echo '<a href="?pagina='.$i.'&search_rut='.$search.'" class="page-link '.$active.'">'.$i.'</a>';
            }
            
            if($pagina < $total_paginas) echo '<a href="?pagina='.($pagina+1).'&search_rut='.$search.'" class="page-link"><i class="fas fa-chevron-right"></i></a>';
            ?>
        </div>
        <?php endif; ?>
    </div>
</main>

</body>
</html>