<?php
// 1. Configuración de seguridad de sesión (DEBE IR PRIMERO)
// Esto asegura que la configuración se aplique antes de que cualquier include inicie la sesión.
ini_set('session.cookie_httponly', 1);

header('Content-Type: text/html; charset=utf-8');

// 2. Includes (Es probable que auth.php inicie la sesión, por eso va después del ini_set)
include 'includes/db.php';
include 'includes/auth.php';

// 3. Verificación de sesión (Por seguridad, por si auth.php no la inició)
if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}

// Inicializar contador de intentos si no existe
if (!isset($_SESSION['intentos_fallidos'])) {
    $_SESSION['intentos_fallidos'] = 0;
    $_SESSION['ultimo_intento'] = time();
}

// Configuración de bloqueo
$max_intentos = 3;
$bloqueo_segundos = 30;

// Verificar si está bloqueado
if ($_SESSION['intentos_fallidos'] >= $max_intentos && (time() - $_SESSION['ultimo_intento'] <= $bloqueo_segundos)) {
    $bloqueado = true;
    $tiempo_restante = $bloqueo_segundos - (time() - $_SESSION['ultimo_intento']);
} else {
    $bloqueado = false;
    // Si ya pasó el tiempo de bloqueo, reiniciar contador
    if ($_SESSION['intentos_fallidos'] >= $max_intentos) {
        $_SESSION['intentos_fallidos'] = 0;
    }
}

$conn->set_charset("utf8");

if ($_SERVER['REQUEST_METHOD'] == 'POST' && !$bloqueado) {
    if (isset($_POST['rut']) && isset($_POST['contrasena'])) {
        $rut = $conn->real_escape_string($_POST['rut']);
        $contrasena = $_POST['contrasena'];

        $stmt = $conn->prepare("SELECT u.*, r.nombre AS rol_nombre FROM usuario u 
                                JOIN roles r ON u.rol_id = r.id WHERE u.rut = ?");
        $stmt->bind_param("s", $rut);
        $stmt->execute();
        $result = $stmt->get_result();

        $login_exitoso = false;
        $user = null;

        // Verificamos si existe el usuario
        if ($result->num_rows == 1) {
            $user_temp = $result->fetch_assoc();
            // Verificamos la contraseña
            if (password_verify($contrasena, $user_temp['contraseña'])) {
                $login_exitoso = true;
                $user = $user_temp;
            }
        }

        if ($login_exitoso) {
            // --- LOGIN CORRECTO ---
            session_regenerate_id(true); // Prevenir fijación de sesión
            $_SESSION['usuario_id'] = $user['id'];
            $_SESSION['nombre'] = $user['nombre'];
            $_SESSION['apellido_paterno'] = $user['apellido_paterno'];
            $_SESSION['apellido_materno'] = $user['apellido_materno'];
            $_SESSION['rol'] = $user['rol_id'];
            $_SESSION['rol_nombre'] = $user['rol_nombre'];
            $_SESSION['usuario'] = $user['nombre'] . ' ' . $user['apellido_paterno'] . ' ' . $user['apellido_materno'];
            
            // Reiniciar intentos
            $_SESSION['intentos_fallidos'] = 0;
            
            header("Location: index.php");
            exit();
        } else {
            // --- LOGIN FALLIDO (Genérico) ---
            $_SESSION['intentos_fallidos']++;
            $_SESSION['ultimo_intento'] = time();
            // Mensaje genérico para no revelar si el RUT existe o no
            $error = "RUT o contraseña incorrectos.";
        }

        $stmt->close();
    } else {
        $error = "Por favor, complete todos los campos.";
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Inicio de Sesión - Escuela Porongo</title>
    <link rel="icon" href="images/logo.png">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;600&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }

        body {
            font-family: 'Poppins', sans-serif;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: linear-gradient(rgba(0,0,0,0.4), rgba(0,0,0,0.6)), url('images/ESCUELA.jpg') no-repeat center center/cover;
        }

        .login-container {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            box-shadow: 0 15px 35px rgba(0,0,0,0.3);
            padding: 45px 40px;
            width: 100%;
            max-width: 420px;
            text-align: center;
            color: #333;
            animation: fadeIn 0.9s ease;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-15px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .login-container img.insignia {
            width: 220px;
            margin-bottom: 15px;
        }

        .login-container h2 {
            color: #007BFF;
            font-weight: 600;
            font-size: 22px;
        }

        .login-container h1 {
            font-size: 26px;
            margin-bottom: 25px;
            color: #222;
        }

        .login-container input {
            width: 100%;
            padding: 12px 15px;
            margin: 12px 0;
            border: 1px solid #ccc;
            border-radius: 8px;
            font-size: 15px;
            transition: all 0.3s;
        }

        .login-container input:focus {
            border-color: #007BFF;
            box-shadow: 0 0 8px rgba(0,123,255,0.3);
            outline: none;
        }

        .login-container button {
            width: 100%;
            padding: 12px;
            background: linear-gradient(135deg, #007BFF, #0056b3);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            margin-top: 10px;
            transition: all 0.3s;
        }

        .login-container button:hover {
            background: linear-gradient(135deg, #0056b3, #003f7f);
            transform: scale(1.03);
            box-shadow: 0 6px 20px rgba(0,91,179,0.4);
        }

        .password-wrapper {
            position: relative;
        }

        .toggle-password {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #007BFF;
            cursor: pointer;
            font-size: 18px;
        }

        .error {
            background: #ffe6e6;
            border-left: 4px solid #ff4c4c;
            color: #b30000;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 15px;
            font-size: 14px;
        }

        .blocked {
            background: #fff3cd;
            border-left: 4px solid #ffc107;
            color: #856404;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 15px;
            font-size: 14px;
        }

        .footer-text {
            font-size: 13px;
            color: #555;
            margin-top: 15px;
            opacity: 0.8;
        }
    </style>
    <script>
        function togglePassword() {
            const passwordField = document.getElementById('password');
            const icon = document.getElementById('toggleIcon');
            const isPassword = passwordField.type === 'password';
            passwordField.type = isPassword ? 'text' : 'password';
            icon.classList.toggle('fa-eye');
            icon.classList.toggle('fa-eye-slash');
        }
    </script>
</head>
<body>
    <div class="login-container">
        <img src="images/INSIGNIA%20SF.png" alt="Insignia Colegio" class="insignia">
        <h2>Escuela Porongo</h2>
        <h1>Inicio de Sesión</h1>

        <?php if (isset($error)): ?>
            <div class="error"><i class="fas fa-exclamation-circle"></i> <?= htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></div>
        <?php endif; ?>

        <?php if ($bloqueado): ?>
            <div class="blocked"><i class="fas fa-lock"></i> Demasiados intentos. Espera <?= $tiempo_restante; ?> segundos.</div>
        <?php else: ?>
            <form method="POST" action="login.php">
                <input type="text" name="rut" placeholder="RUT (Ej: 12345678-9)" required autocomplete="username">
                <div class="password-wrapper">
                    <input type="password" name="contrasena" placeholder="Contraseña" id="password" required autocomplete="current-password">
                    <span class="toggle-password" onclick="togglePassword()">
                        <i id="toggleIcon" class="fas fa-eye"></i>
                    </span>
                </div>
                <button type="submit">Ingresar</button>
            </form>
        <?php endif; ?>

        <div class="footer-text">© 2024 Escuela Porongo - Todos los derechos reservados</div>
    </div>
</body>
</html>