<?php
session_start(); // Iniciar la sesión
header('Content-Type: text/html; charset=utf-8');
include '../includes/db.php';
include '../includes/auth.php';
checkAuth();

$usuarioSesion = $_SESSION['usuario_id'] ?? null;
if(!$usuarioSesion){
    die("No se ha detectado usuario en sesión.");
}

// AÑADIDO: Manejo de mensajes de sesión
$message = '';
$message_type = '';
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message'], $_SESSION['message_type']);
}

// MODIFICADO: Eliminar log (con redirección y mensaje de sesión)
if (isset($_GET['eliminar_log_id'])) {
    $id = intval($_GET['eliminar_log_id']);
    $stmt = $conn->prepare("DELETE FROM justificativos_log WHERE id = ?");
    $stmt->bind_param("i", $id);
    
    if ($stmt->execute()) {
        $_SESSION['message'] = 'Registro de log eliminado exitosamente.';
        $_SESSION['message_type'] = 'exito';
    } else {
        $_SESSION['message'] = 'Error al eliminar el log: ' . $stmt->error;
        $_SESSION['message_type'] = 'error';
    }
    
    $stmt->close();
    header("Location: logs_justificativos.php");
    exit;
}

// Paginación
$registros_por_pagina = 10;
$pagina_actual = isset($_GET['pagina']) ? max(1, intval($_GET['pagina'])) : 1;
$inicio = ($pagina_actual - 1) * $registros_por_pagina;

// Total de registros
$result_total = $conn->query("SELECT COUNT(*) AS total FROM justificativos_log");
$total_registros = $result_total->fetch_assoc()['total'];
$total_paginas = ceil($total_registros / $registros_por_pagina);

// Obtener logs (tu consulta JOIN es correcta)
$logs = $conn->query("
    SELECT jl.*, 
           u.nombre AS usuario_nombre, u.apellido_paterno AS usuario_ap, u.apellido_materno AS usuario_am,
           a.nombre AS alumno_nombre, a.apellido_paterno AS alumno_ap_alumno, a.apellido_materno AS alumno_am_alumno,
           c.nombre AS curso_nombre
    FROM justificativos_log jl
    LEFT JOIN usuario u ON jl.usuario_id = u.id
    LEFT JOIN justificativos j ON jl.justificativo_id = j.id
    LEFT JOIN usuario a ON j.usuario_id = a.id
    LEFT JOIN curso c ON j.curso_id = c.id
    ORDER BY jl.fecha DESC
    LIMIT $inicio, $registros_por_pagina
");
?>

<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<link rel="icon" type="image/png" href="../images/logo.png">
<title>Logs de Justificativos</title>

<link rel="stylesheet" href="../css/admin.css">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">

<style>
    .log-grid-container {
        display: grid;
        /* Se ajusta automáticamente, mínimo 350px, máximo 1fr (2 columnas en pantallas medias) */
        grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
        gap: 1.5rem;
    }
    .log-card {
        background-color: #fff;
        border-radius: var(--border-radius);
        box-shadow: var(--box-shadow);
        padding: 1.5rem;
        display: flex;
        flex-direction: column;
    }
    .log-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 1rem;
        border-bottom: 1px solid var(--grey-light);
        padding-bottom: 0.75rem;
    }
    .log-header .date {
        font-size: 0.9rem;
        color: var(--grey-dark);
        font-weight: 500;
    }
    .log-badge {
        padding: 0.25rem 0.75rem;
        border-radius: 12px;
        font-weight: 700;
        font-size: 0.8rem;
        color: var(--light-color);
        background-color: var(--primary-color); /* Azul por defecto */
    }
    .log-badge.eliminar { background-color: var(--danger-color); }
    .log-badge.crear { background-color: var(--secondary-color); }
    .log-badge.modificar { background-color: var(--warning-color); }
    
    .log-body p {
        margin-bottom: 0.75rem;
        color: var(--grey-dark);
        line-height: 1.4;
    }
    .log-body p strong {
        color: var(--dark-color);
        width: 80px; /* Alinea los labels */
        display: inline-block;
    }
    .log-body .log-detail {
        background-color: var(--grey-light);
        padding: 0.75rem;
        border-radius: var(--border-radius);
        font-size: 0.9rem;
        white-space: pre-wrap; /* Mantiene saltos de línea */
        word-break: break-word;
    }
</style>
</head>
<body>

<header>
    <div class="header-content">
        <img src="../images/INSIGNIA SF.png" alt="Logo">
        <h1>Logs de Justificativos</h1>
        <nav>
            <ul>
                <li><a href="../index.php"><i class="fas fa-home"></i> Inicio</a></li>
                <li><a href="gestion_justificativos.php"><i class="fas fa-arrow-left"></i> Volver</a></li>
            </ul>
        </nav>
    </div>
</header>

<main class="container">

    <?php if ($message): ?>
        <div class="mensaje <?php echo $message_type; ?> show">
            <?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <div class="card table-container">
        <h2><i class="fas fa-history"></i> Registro de Actividad</h2>

        <div class="log-grid-container">
            <?php if ($logs->num_rows > 0): ?>
                <?php while($log = $logs->fetch_assoc()): ?>
                    <div class="log-card">
                        <div class="log-header">
                            <span class="date"><i class="fas fa-clock"></i> <?= date('d-m-Y H:i', strtotime($log['fecha'])) ?></span>
                            <div>
                                <span class="log-badge <?= strtolower($log['accion']) ?>">
                                    <?= htmlspecialchars($log['accion']) ?>
                                </span>
                                <a href="?eliminar_log_id=<?= $log['id'] ?>" 
                                   onclick="return confirm('¿Seguro que deseas eliminar este log?');" 
                                   class="btn btn-danger btn-sm" title="Eliminar Log" style="margin-left: 0.5rem;">
                                   <i class="fas fa-trash-alt"></i>
                                </a>
                            </div>
                        </div>
                        <div class="log-body">
                            <p><strong>Usuario:</strong> <?= htmlspecialchars($log['usuario_ap'].' '.$log['usuario_am'].', '.$log['usuario_nombre']) ?></p>
                            <p><strong>Alumno:</strong> <?= htmlspecialchars($log['alumno_ap_alumno'].' '.$log['alumno_am_alumno'].', '.$log['alumno_nombre']) ?></p>
                            <p><strong>Curso:</strong> <?= htmlspecialchars($log['curso_nombre'] ?? 'Desconocido') ?></p>
                            <p><strong>Detalle:</strong></p>
                            <div class="log-detail"><?= nl2br(htmlspecialchars($log['detalle'])) ?></div>
                        </div>
                    </div>
                <?php endwhile; ?>
            <?php else: ?>
                <p style="text-align: center; grid-column: 1 / -1; color: var(--grey-dark); padding: 2rem;">
                    No hay registros de actividad.
                </p>
            <?php endif; ?>
        </div>
    </div>

    <div class="pagination">
        <?php for($i=1; $i<=$total_paginas; $i++): ?>
            <a href="?pagina=<?= $i ?>" class="<?= ($i==$pagina_actual)?'current-page':'' ?>"><?= $i ?></a>
        <?php endfor; ?>
    </div>
</main>

<script>
window.onload = function() {
    const message = document.querySelector('.mensaje');
    if (message) {
        setTimeout(() => {
            message.classList.remove('show');
        }, 3000);
    }
};
</script>
</body>
</html>