<?php
session_start();
header('Content-Type: text/html; charset=utf-8');

include '../includes/db.php';
include '../includes/auth.php';
checkAuth();

$usuarioSesion = $_SESSION['usuario_id'];
date_default_timezone_set('America/Santiago'); 

// --- Lógica de Filtros (Se mantiene igual) ---
$filtro_rut = trim($_GET['filtro_rut'] ?? '');
$filtro_nombre = trim($_GET['filtro_nombre'] ?? '');
$filtro_fecha = trim($_GET['filtro_fecha'] ?? '');

$where_conditions = [];
$params = [];
$types = "";

if ($filtro_rut !== '') {
    $where_conditions[] = "j.rut_retirante LIKE ?";
    $params[] = "%$filtro_rut%";
    $types .= "s";
}
if ($filtro_nombre !== '') {
    $where_conditions[] = "(u.nombre LIKE ? OR u.apellido_paterno LIKE ? OR u.apellido_materno LIKE ?)";
    $params[] = "%$filtro_nombre%";
    $params[] = "%$filtro_nombre%";
    $params[] = "%$filtro_nombre%";
    $types .= "sss";
}
if ($filtro_fecha !== '') {
    $where_conditions[] = "DATE(j.fecha) = ?";
    $params[] = $filtro_fecha;
    $types .= "s";
}

$whereSQL = "";
if (!empty($where_conditions)) {
    $whereSQL = "WHERE " . implode(" AND ", $where_conditions);
}

$filter_query_string = http_build_query(array_filter([
    'filtro_rut' => $filtro_rut,
    'filtro_nombre' => $filtro_nombre,
    'filtro_fecha' => $filtro_fecha
]));

// Manejo de mensajes
$message = '';
$message_type = '';
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message'], $_SESSION['message_type']);
}

// === Lógica de registro (POST) ===
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['registrar'])) {
    
    $tipo_justificacion = $_POST['tipo_justificacion'] ?? 'alumno'; 
    $curso_id = intval($_POST['curso_id'] ?? 0);
    $motivo = $_POST['motivo'] ?? '';
    $nombre_retirante = $_POST['nombre_retirante'] ?? '';
    $rut_retirante = $_POST['rut_retirante'] ?? '';
    $observaciones = $_POST['observaciones'] ?? '';
    $fecha_input = $_POST['fecha_registro'] ?? date('Y-m-d');
    $hora_input = $_POST['hora_registro'] ?? date('H:i');
    $tipo_retirante = $_POST['tipo_retirante'] ?? ''; 

    $usuario_id_registro = 0; 

    if ($tipo_justificacion === 'alumno') {
        $usuario_id_registro = intval($_POST['alumno_id'] ?? 0);
    } elseif ($tipo_justificacion === 'curso') {
        // En caso de curso, usamos el docente seleccionado como "alumno" (responsable)
        // Si usamos el selector de arriba para llenar los datos, necesitamos saber el ID para guardar
        // Sin embargo, para el registro "curso completo", lo importante es el curso_id y el rol.
        // Usaremos un valor placeholder o el ID del docente si se envió.
        $usuario_id_registro = intval($_POST['docente_retirante_select'] ?? 0); 
        // Si no se seleccionó en el dropdown específico pero se llenó nombre, 
        // buscamos si existe un usuario con ese nombre (opcional), o guardamos 0.
    }

    // Validaciones
    if($curso_id === 0) {
        $_SESSION['message'] = "❌ Debes seleccionar un curso.";
        $_SESSION['message_type'] = "error";
    } elseif ($tipo_justificacion === 'alumno' && $usuario_id_registro === 0) {
        $_SESSION['message'] = "❌ Debes seleccionar un alumno.";
        $_SESSION['message_type'] = "error";
    } else {
        
        $fecha_completa = $fecha_input . ' ' . $hora_input . ':00';

        $stmt = $conn->prepare("INSERT INTO justificativos (usuario_id, curso_id, motivo, tipo_retirante, nombre_retirante, rut_retirante, observaciones, usuario_registro, fecha) 
                                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        
        $stmt->bind_param("iisssssis", $usuario_id_registro, $curso_id, $motivo, $tipo_retirante, $nombre_retirante, $rut_retirante, $observaciones, $usuarioSesion, $fecha_completa);

        if($stmt->execute()) {
            // Obtener nombres para mensaje
            $nombre_curso = 'ID: ' . $curso_id;
            $stmt_curso = $conn->prepare("SELECT nombre FROM curso WHERE id = ?");
            $stmt_curso->bind_param("i", $curso_id);
            $stmt_curso->execute();
            $res = $stmt_curso->get_result();
            if ($r = $res->fetch_assoc()) $nombre_curso = htmlspecialchars($r['nombre']);
            $stmt_curso->close();

            $nombre_justificado = '';
            if ($usuario_id_registro > 0) {
                $stmt_name = $conn->prepare("SELECT nombre, apellido_paterno FROM usuario WHERE id = ?");
                $stmt_name->bind_param("i", $usuario_id_registro);
                $stmt_name->execute();
                $res_name = $stmt_name->get_result();
                if ($rn = $res_name->fetch_assoc()) $nombre_justificado = trim($rn['nombre'] . ' ' . $rn['apellido_paterno']);
                $stmt_name->close();
            }

            if ($tipo_justificacion === 'curso') {
                $_SESSION['message'] = "✅ Justificativo de CURSO COMPLETO registrado correctamente en curso: **$nombre_curso**.";
                $detalle_log = "Justificativo CURSO COMPLETO. Curso: $nombre_curso, Retirado por: $nombre_retirante";
            } else {
                $_SESSION['message'] = "✅ Justificativo de ALUMNO ($nombre_justificado) registrado.";
                $detalle_log = "Justificativo ALUMNO. Alumno: $nombre_justificado, Curso: $nombre_curso, Retirado por: $nombre_retirante";
            }
            $_SESSION['message_type'] = "exito";
            
            $justificativo_id = $stmt->insert_id;
            $log = $conn->prepare("INSERT INTO justificativos_log (justificativo_id, usuario_id, accion, detalle) VALUES (?, ?, 'CREAR', ?)");
            $log->bind_param("iis", $justificativo_id, $usuarioSesion, $detalle_log);
            $log->execute(); $log->close();

        } else {
            $_SESSION['message'] = "❌ Error: " . $stmt->error;
            $_SESSION['message_type'] = "error";
        }
        $stmt->close();
    }
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Datos
$cursos = $conn->query("SELECT id, nombre FROM curso WHERE activo = 1 ORDER BY nombre ASC");
$docentes_result = $conn->query("SELECT id, nombre, apellido_paterno, apellido_materno, rut FROM usuario WHERE rol_id = 2 AND activo = 1 ORDER BY apellido_paterno");
$docentes = [];
while ($d = $docentes_result->fetch_assoc()) { $docentes[] = $d; }

$alumnos_result = $conn->query("SELECT id, nombre, apellido_paterno, apellido_materno, curso_id FROM usuario WHERE rol_id = 3 AND activo = 1 ORDER BY apellido_paterno");
$alumnos = [];
while ($a = $alumnos_result->fetch_assoc()) { $alumnos[] = $a; }

// Paginación
$registros_por_pagina = 10;
$pagina_actual = isset($_GET['pagina']) ? max(1, intval($_GET['pagina'])) : 1;
$inicio = ($pagina_actual - 1) * $registros_por_pagina;

$sqlTotal = "SELECT COUNT(j.id) AS total FROM justificativos j LEFT JOIN usuario u ON j.usuario_id = u.id $whereSQL";
$stmtTotal = $conn->prepare($sqlTotal);
if (!empty($params)) $stmtTotal->bind_param($types, ...$params);
$stmtTotal->execute();
$total_registros = $stmtTotal->get_result()->fetch_assoc()['total'];
$total_paginas = ceil($total_registros / $registros_por_pagina);
$stmtTotal->close();

$sqlJustificativos = "
    SELECT j.*, r.nombre AS rol_usuario, u.rol_id AS rol_id, u.nombre AS usuario_nombre, u.apellido_paterno AS usuario_ap, u.apellido_materno AS usuario_am,
           c.nombre AS curso_nombre, f.nombre AS func_nombre, f.apellido_paterno AS func_ap, f.apellido_materno AS func_am
    FROM justificativos j
    INNER JOIN usuario u ON j.usuario_id = u.id 
    INNER JOIN roles r ON u.rol_id = r.id 
    LEFT JOIN curso c ON j.curso_id = c.id
    LEFT JOIN usuario f ON j.usuario_registro = f.id
    $whereSQL ORDER BY j.fecha DESC LIMIT ?, ?";
$stmtJustificativos = $conn->prepare($sqlJustificativos);
$all_params = array_merge($params, [$inicio, $registros_por_pagina]);
$all_types = $types . "ii";
$bind_params = []; foreach ($all_params as $k => $v) $bind_params[$k] = &$all_params[$k];
if (!empty($all_params)) call_user_func_array([$stmtJustificativos, 'bind_param'], array_merge([$all_types], $bind_params));
$stmtJustificativos->execute();
$justificativos = $stmtJustificativos->get_result();
?>

<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<link rel="icon" type="image/png" href="../images/logo.png">
<title>Registro de Justificativos</title>
<link rel="stylesheet" href="../css/admin.css">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
<link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
<style>
    .form-grid-3col { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1.5rem; }
    .switch-field { display: flex; justify-content: center; margin-bottom: 1.5rem; background: var(--grey-light); border-radius: 6px; padding: 4px; }
    .switch-field input { position: absolute !important; clip: rect(1px, 1px, 1px, 1px); height: 1px; width: 1px; overflow: hidden; }
    .switch-field label { background-color: transparent; color: var(--text-color); font-size: 14px; text-align: center; padding: 10px 15px; margin: 0; cursor: pointer; flex-grow: 1; border-radius: 4px; font-weight: 500; transition: 0.2s; }
    .switch-field input:checked + label { background-color: var(--primary-color); color: white; box-shadow: 0 2px 4px 0 rgba(0,0,0,0.1); }
    #alumno_id option[data-curso] { display: none; }
    .justificativo-curso { background-color: #fff3cd; border-left: 5px solid #ffc107; }
</style>
</head>
<body>

<header>
    <div class="header-content">
        <img src="../images/INSIGNIA SF.png" alt="Insignia Escuela">
        <h1>Registro de Justificativos</h1>
        <nav>
            <ul>
                <li><a href="../index.php"><i class="fas fa-home"></i> Inicio</a></li>
                <li><a href="gestion_justificativos.php"><i class="fas fa-list-check"></i> Gestionar</a></li>
            </ul>
        </nav>
    </div>
</header>

<main class="container">
    <?php if ($message): ?><div class="mensaje <?php echo $message_type; ?> show"><?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?></div><?php endif; ?>

    <div class="card form-container">
        <h2><i class="fas fa-file-signature"></i> Registrar Nuevo Justificativo</h2>
        <form method="POST">
            
            <div class="switch-field">
                <input type="radio" id="radio-alumno" name="tipo_justificacion" value="alumno" checked/>
                <label for="radio-alumno"><i class="fas fa-user-graduate"></i> Alumno Individual</label>
                <input type="radio" id="radio-curso" name="tipo_justificacion" value="curso" />
                <label for="radio-curso"><i class="fas fa-users"></i> Curso Completo</label>
            </div>

            <div class="form-grid">
                <div class="form-group">
                    <label><i class="fas fa-calendar-day"></i> Fecha</label>
                    <input type="date" name="fecha_registro" value="<?= date('Y-m-d') ?>" required>
                </div>
                <div class="form-group">
                    <label><i class="fas fa-clock"></i> Hora</label>
                    <input type="time" name="hora_registro" value="<?= date('H:i') ?>" required>
                </div>
            </div>
            
            <hr style="margin: 1.5rem 0; border: 0; border-top: 1px solid var(--grey-light);">
            
            <h3 style="margin-bottom: 1rem; color: var(--primary-color); font-size: 1.1rem;">Datos del Retirante</h3>
            <div class="form-grid-3col">
                <div class="form-group">
                    <label><i class="fas fa-user-check"></i> Tipo de Retirante</label>
                    <select name="tipo_retirante" id="tipo_retirante" required> 
                        <option value="Apoderado Titular">Apoderado Titular</option>
                        <option value="Apoderado Suplente">Apoderado Suplente</option>
                        <option value="Docente">Docente</option>
                        <option value="Otro">Otro</option>
                    </select>
                </div>
                
                <div class="form-group" id="retirante_docente_group" style="display:none;">
                    <label><i class="fas fa-chalkboard-teacher"></i> Seleccionar Docente</label>
                    <select name="docente_retirante_select" id="docente_retirante_select">
                        <option value="">Seleccionar...</option>
                        <?php foreach($docentes as $d): 
                            $nombre_completo = htmlspecialchars($d['apellido_paterno'].' '.$d['apellido_materno'].', '.$d['nombre']);
                            $rut_docente = htmlspecialchars($d['rut']); ?>
                            <option value="<?= $d['id'] ?>" data-rut="<?= $rut_docente ?>"><?= $nombre_completo ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label><i class="fas fa-user"></i> Nombre Retirante</label>
                    <input type="text" id="nombre_retirante" name="nombre_retirante" required placeholder="Nombre del retirante">
                </div>
                <div class="form-group">
                    <label><i class="fas fa-id-card"></i> RUT Retirante</label>
                    <input type="text" id="rut_retirante" name="rut_retirante" placeholder="RUT del retirante">
                </div>
            </div>
            
            <hr style="margin: 1.5rem 0; border: 0; border-top: 1px solid var(--grey-light);">

            <h3 style="margin-bottom: 1rem; color: var(--primary-color); font-size: 1.1rem;">Datos del Retiro</h3>
            <div class="form-grid">
                <div class="form-group">
                    <label><i class="fas fa-university"></i> Curso</label>
                    <select name="curso_id" id="curso_id" required>
                        <option value="">Seleccionar curso</option>
                        <?php $cursos->data_seek(0); while($c = $cursos->fetch_assoc()): ?>
                            <option value="<?= $c['id'] ?>"><?= htmlspecialchars($c['nombre']) ?></option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div class="form-group" id="alumno_select_group">
                    <label><i class="fas fa-user-graduate"></i> Alumno</label>
                    <select name="alumno_id" id="alumno_id"> 
                        <option value="">Seleccionar alumno</option>
                        <?php foreach($alumnos as $a): ?>
                            <option value="<?= $a['id'] ?>" data-curso="<?= $a['curso_id'] ?>"><?= htmlspecialchars($a['apellido_paterno'].' '.$a['nombre']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group" id="curso_completo_msg" style="display:none; align-items: center; color: var(--warning-color); font-weight: bold;">
                    <i class="fas fa-users" style="font-size: 1.5rem; margin-right: 10px;"></i> 
                    Se justificará a todo el curso seleccionado.
                </div>
            </div>
            
            <div class="form-group full-width">
                <label><i class="fas fa-comment-dots"></i> Motivo</label>
                <textarea name="motivo" rows="2" required></textarea>
            </div>
            
            <div class="form-group full-width">
                <label><i class="fas fa-eye"></i> Observaciones</label>
                <textarea name="observaciones" rows="2"></textarea>
            </div>
            
            <div class="form-buttons">
                <button type="submit" name="registrar" class="btn btn-primary"><i class="fas fa-save"></i> Registrar</button>
            </div>
        </form>
    </div>

    <div class="card table-container" style="margin-top: 2rem;">
        <h2><i class="fas fa-history"></i> Últimos Registros</h2>
        <div class="table-responsive">
            <table>
                <thead><tr><th>Fecha</th><th>Justificado</th><th>Curso</th><th>Retirante</th><th>Motivo</th><th>Funcionario</th></tr></thead>
                <tbody>
                    <?php if ($justificativos->num_rows > 0): ?>
                        <?php while($j = $justificativos->fetch_assoc()): 
                             $is_docente = ($j['rol_id'] == 2); 
                             $nombre = htmlspecialchars($j['usuario_ap'].' '.$j['usuario_nombre']);
                        ?>
                            <tr class="<?= $is_docente ? 'justificativo-curso' : '' ?>">
                                <td><?= date('d-m H:i', strtotime($j['fecha'])) ?></td>
                                <td><?= $is_docente ? '<strong>CURSO</strong>' : $nombre ?></td>
                                <td><?= htmlspecialchars($j['curso_nombre'] ?? 'N/A') ?></td> 
                                <td><?= htmlspecialchars($j['nombre_retirante']) ?></td>
                                <td><?= htmlspecialchars($j['motivo']) ?></td>
                                <td><?= htmlspecialchars($j['func_nombre'].' '.$j['func_ap'].' '.$j['func_am']) ?></td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr><td colspan="5" style="text-align: center;">No hay registros recientes.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</main>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const message = document.querySelector('.mensaje');
        if (message) setTimeout(() => { message.classList.remove('show'); }, 3000);

        const cursoSelect = document.getElementById('curso_id');
        const radioAlumno = document.getElementById('radio-alumno');
        const radioCurso = document.getElementById('radio-curso');
        
        const alumnoGroup = document.getElementById('alumno_select_group');
        const alumnoSelect = document.getElementById('alumno_id');
        const alumnoOptions = Array.from(alumnoSelect.querySelectorAll('option[data-curso]'));
        const cursoCompletoMsg = document.getElementById('curso_completo_msg');
        
        // Campos Retirante
        const tipoRetiranteSelect = document.getElementById('tipo_retirante');
        const nombreRetiranteInput = document.getElementById('nombre_retirante');
        const rutRetiranteInput = document.getElementById('rut_retirante');
        
        const retiranteDocenteGroup = document.getElementById('retirante_docente_group');
        const retiranteDocenteSelect = document.getElementById('docente_retirante_select');
        
        const defaultPlaceholderNombre = nombreRetiranteInput.placeholder;
        const defaultPlaceholderRut = rutRetiranteInput.placeholder;

        // 1. LÓGICA DE TIPO DE RETIRANTE
        function handleTipoRetirante() {
            if (tipoRetiranteSelect.value === 'Docente') {
                // Mostrar selector de docentes
                retiranteDocenteGroup.style.display = 'block';
                nombreRetiranteInput.readOnly = true; // Opcional: bloquear manual si seleccionas de lista
                rutRetiranteInput.readOnly = true;
                
                // Si ya había uno seleccionado, rellenar
                if (retiranteDocenteSelect.value) {
                    fillDocenteData();
                } else {
                    nombreRetiranteInput.value = '';
                    rutRetiranteInput.value = '';
                }
            } else {
                // Ocultar selector y limpiar/habilitar campos manuales
                retiranteDocenteGroup.style.display = 'none';
                nombreRetiranteInput.readOnly = false;
                rutRetiranteInput.readOnly = false;
                nombreRetiranteInput.value = '';
                rutRetiranteInput.value = '';
                retiranteDocenteSelect.value = '';
            }
        }

        function fillDocenteData() {
            const opt = retiranteDocenteSelect.options[retiranteDocenteSelect.selectedIndex];
            if (opt.value) {
                nombreRetiranteInput.value = opt.text;
                rutRetiranteInput.value = opt.dataset.rut || '';
            } else {
                nombreRetiranteInput.value = '';
                rutRetiranteInput.value = '';
            }
        }
        
        tipoRetiranteSelect.addEventListener('change', handleTipoRetirante);
        retiranteDocenteSelect.addEventListener('change', fillDocenteData);

        // 2. FILTRO DE ALUMNOS POR CURSO
        function filterAlumnos() {
            var cursoId = cursoSelect.value;
            alumnoSelect.value = '';
            alumnoOptions.forEach(function(option){
                option.style.display = (option.dataset.curso === cursoId || option.value === "") ? 'block' : 'none';
            });
        }
        cursoSelect.addEventListener('change', filterAlumnos);

        // 3. SWITCH MODO (Alumno vs Curso)
        function toggleJustificacionTipo() {
            if (radioCurso.checked) {
                // MODO CURSO COMPLETO
                alumnoGroup.style.display = 'none';
                cursoCompletoMsg.style.display = 'flex';
                alumnoSelect.removeAttribute('required');
                
                // Forzar tipo retirante a Docente
                tipoRetiranteSelect.value = 'Docente';
                handleTipoRetirante();

            } else {
                // MODO ALUMNO INDIVIDUAL
                alumnoGroup.style.display = 'block';
                cursoCompletoMsg.style.display = 'none';
                alumnoSelect.setAttribute('required', 'required');
                
                // Restaurar tipo a Apoderado (por defecto)
                tipoRetiranteSelect.value = 'Apoderado Titular';
                handleTipoRetirante();
            }
            filterAlumnos();
        }

        radioAlumno.addEventListener('change', toggleJustificacionTipo);
        radioCurso.addEventListener('change', toggleJustificacionTipo);

        toggleJustificacionTipo(); // Init
    });
</script>
</body>
</html>