<?php
session_start();
header('Content-Type: text/html; charset=utf-8');
include 'includes/db.php';
include 'includes/auth.php';
checkAuth();

$conn->set_charset("utf8");

// --- Control de sesión por inactividad ---
$limiteInactividad = 30 * 60; 
if (isset($_SESSION['ultimoAcceso'])) {
    $inactividad = time() - $_SESSION['ultimoAcceso'];
    if ($inactividad > $limiteInactividad) {
        session_unset();
        session_destroy();
        header("Location: login.php");
        exit();
    }
}
$_SESSION['ultimoAcceso'] = time();

// --- Carga de permisos ---
$permisos_usuario = [];
$rol_id = $_SESSION['rol'] ?? null;
$usuario_id = $_SESSION['usuario_id'] ?? null;

if ($rol_id || $usuario_id) {
    // Nota: Quitamos el ORDER BY de SQL porque ordenaremos con PHP abajo
    $stmt_rol = $conn->prepare("SELECT p.nombre, p.url FROM permisos p JOIN roles_permisos rp ON p.id = rp.permiso_id WHERE rp.rol_id = ?");
    $stmt_rol->bind_param('i', $rol_id);
    $stmt_rol->execute();
    $result_rol = $stmt_rol->get_result();
    while ($row = $result_rol->fetch_assoc()) { $permisos_usuario[$row['url']] = $row['nombre']; }
    $stmt_rol->close();

    $stmt_usuario = $conn->prepare("SELECT p.nombre, p.url FROM permisos p JOIN usuario_permisos up ON p.id = up.permiso_id WHERE up.usuario_id = ?");
    $stmt_usuario->bind_param('i', $usuario_id);
    $stmt_usuario->execute();
    $result_usuario = $stmt_usuario->get_result();
    while ($row = $result_usuario->fetch_assoc()) { $permisos_usuario[$row['url']] = $row['nombre']; }
    $stmt_usuario->close();
}

// =========================================================
// --- ORDEN PERSONALIZADO (AQUÍ DEFINES TU ORDEN) ---
// =========================================================
$orden_prioridad = [
    // Pon aquí los nombres EXACTOS de tus permisos en el orden que desees
    "Administrar Usuarios",
    "Gestionar Roles",
    "Agendar Horas",
    "Generar Informes de Notas",
    "Ver Notas",
    "Agregar Notas",
    "Gestionar Notas",
    "Generar Certificado de Matrícula",
    "Generar Certificado Alumno Regular",
    "Generar Certificado Traslado",
    "Ver Cursos y Alumnos",
    "Agregar Curso",
    "Agregar Asignaturas",
    "Agregar Alumnos",
    "Formulario de Matrícula",
    "Gestionar Matrículas",
    "Respaldo y Recuperación",
    "Prestamos",
    "Ingresar Equipos",
    "Gestionar Equipos",
    "Permisos",
    "Justificativos",
    "Horas Docentes",
    // Los permisos que NO pongas aquí saldrán al final
];

// Función para ordenar el array de permisos según la lista de arriba
uasort($permisos_usuario, function($a, $b) use ($orden_prioridad) {
    $pos_a = array_search($a, $orden_prioridad);
    $pos_b = array_search($b, $orden_prioridad);

    // Si ambos están en la lista, comparar posiciones
    if ($pos_a !== false && $pos_b !== false) {
        return $pos_a - $pos_b;
    }
    // Si solo $a está en la lista, va antes
    if ($pos_a !== false) return -1;
    // Si solo $b está en la lista, va antes
    if ($pos_b !== false) return 1;
    
    // Si ninguno está en la lista, ordenar alfabéticamente
    return strcasecmp($a, $b);
});
// =========================================================

$iconos_permisos = [
    "Administrar Usuarios" => "&#128100;",
    "Gestionar Roles" => "&#128273;",
    "Generar Informes de Notas" => "&#128200;",
    "Ver Notas" => "&#128221;",
    "Agregar Notas" => "&#10133;",
    "Gestionar Notas" => "&#128202;",
    "Generar Certificado Alumno Regular" => "&#127891;",
    "Generar Certificado Traslado" => "&#127979;",
    "Generar Certificado de Matrícula" => "&#127979;",
    "Ver Cursos y Alumnos" => "&#128218;",
    "Agregar Curso" => "&#128196;",
    "Agregar Asignaturas" => "&#10133;",
    "Agregar Alumnos" => "&#128104;&#8205;&#127891;",
    "Formulario de Matrícula" => "&#128203;",
    "Gestionar Matrículas" => "&#128203;",
    "Respaldo y Recuperación" => "&#128190;",
    "Prestamos" => "&#128230;",
    "Ingresar Equipos" => "&#128187;",
    "Gestionar Equipos" => "&#128187;",
    "Agendar Horas" => "&#9200;",
    "Permisos" => "&#128273;",
    "Justificativos" => "&#128203;",
    "Horas Docentes" => "&#9200;",
];
$icono_default = '&#128197;';

$secciones = [
    'Usuarios' => ['Administrar Usuarios', 'Gestionar Roles', 'Permisos'],
    'Notas y Certificados' => ['Generar Informes de Notas', 'Ver Notas', 'Agregar Notas', 'Gestionar Notas'],
    'Certificados' => ['Generar Certificado Alumno Regular', 'Generar Certificado Traslado'],
    'Cursos y Alumnos' => ['Ver Cursos y Alumnos', 'Agregar Curso', 'Agregar Asignaturas', 'Agregar Alumnos'],
    'Matrículas' => ['Gestionar Matrículas'],
    'Equipos' => ['Gestionar Equipos', 'Prestamos'],
    'Agendamiento' => ['Agendar Horas'],
    'Sistema' => ['Respaldo y Recuperación']
];
$menu_agrupado = [];
$mapa_permisos_seccion = [];
foreach ($secciones as $titulo_seccion => $permisos_en_seccion) {
    foreach ($permisos_en_seccion as $nombre_permiso) {
        $mapa_permisos_seccion[$nombre_permiso] = $titulo_seccion;
    }
}
foreach ($permisos_usuario as $url => $nombre) {
    $titulo_seccion = $mapa_permisos_seccion[$nombre] ?? 'Otros';
    if (!isset($menu_agrupado[$titulo_seccion])) { $menu_agrupado[$titulo_seccion] = []; }
    $menu_agrupado[$titulo_seccion][] = ['url' => $url, 'nombre' => $nombre, 'icono' => $iconos_permisos[$nombre] ?? $icono_default];
}
if (isset($menu_agrupado['Otros'])) {
    $otros = $menu_agrupado['Otros']; unset($menu_agrupado['Otros']); $menu_agrupado['Otros'] = $otros;
}

function getCountByRole($conn, $roleName) {
    $stmt = $conn->prepare("SELECT COUNT(u.id) AS total FROM usuario u JOIN roles r ON u.rol_id = r.id WHERE r.nombre = ? AND u.activo = 1");
    if (!$stmt) return 0;
    $stmt->bind_param("s", $roleName); $stmt->execute();
    return $stmt->get_result()->fetch_assoc()['total'] ?? 0;
}
function getCountFromTable($conn, $tableName) {
     $stmt = $conn->prepare("SELECT COUNT(*) AS total FROM $tableName WHERE activo = 1");
     if (!$stmt) return 0;
     $stmt->execute();
     return $stmt->get_result()->fetch_assoc()['total'] ?? 0;
}

if (isAdmin()) {
    $total_estudiantes = getCountByRole($conn, 'Estudiante');
    $total_docentes = getCountByRole($conn, 'Docente');
    $total_asistentes = getCountByRole($conn, 'Asistente');
    $total_administrativos = getCountByRole($conn, 'Administrativo'); 
    $total_cursos = getCountFromTable($conn, 'curso');
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <link rel="icon" type="image/png" href="images/logo.png">
    <title>Panel Principal - Escuela Porongo</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    
    <style>
        :root {
            --primary: #2563eb; --primary-dark: #1e40af; --sidebar-bg: #0f172a;
            --sidebar-text: #e2e8f0; --bg-body: #f8fafc; --card-bg: #ffffff;
            --text-main: #334155; --text-muted: #64748b;
            --shadow: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --transition: all 0.3s ease;
        }

        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Poppins', sans-serif;
            background-color: var(--bg-body);
            background-image: url('images/COLEGIO.jpg'); 
            background-size: cover; background-attachment: fixed; background-position: center;
            min-height: 100vh; display: flex; color: var(--text-main); overflow-y: scroll; 
        }

        body::before { content: ""; position: fixed; inset: 0; background: rgba(248, 250, 252, 0.85); z-index: -1; }

        /* ---------- SIDEBAR CORREGIDO (Flex Sticky Footer) ---------- */
        .sidebar {
            width: 280px; 
            background: var(--sidebar-bg); 
            color: var(--sidebar-text);
            padding: 20px 16px; 
            
            /* Usamos dvh para mejor soporte en móviles (dynamic viewport height) */
            height: 100dvh; 
            
            position: fixed; top: 0; left: 0;
            box-shadow: 4px 0 15px rgba(0,0,0,0.1);
            transition: transform 0.3s ease; z-index: 1000;
            
            /* NUEVO: Flexbox para estructurar el contenido verticalmente */
            display: flex;
            flex-direction: column;
            justify-content: space-between; /* Logo arriba, Contenido medio, Logout abajo */
        }

        /* Logo fijo arriba */
        .logo-titulo { 
            text-align: center; 
            margin-bottom: 20px; 
            padding-bottom: 15px; 
            border-bottom: 1px solid rgba(255,255,255,0.1); 
            flex-shrink: 0; /* No encoger */
        }
        .logo-titulo img { width: 80px; height: 80px; object-fit: contain; filter: drop-shadow(0 4px 6px rgba(0,0,0,0.3)); }
        .logo-titulo h2 { font-size: 1.1rem; margin-top: 10px; font-weight: 600; color: #fff; letter-spacing: 0.5px; }
        
        /* CONTENEDOR DEL MENÚ SCROLLABLE */
        .menu-scroll-container {
            flex-grow: 1; /* Ocupa todo el espacio disponible en el medio */
            overflow-y: auto; /* El scroll ocurre SOLO aquí dentro */
            margin-bottom: 15px;
            
            /* Scrollbar personalizada fina */
            scrollbar-width: thin;
            scrollbar-color: rgba(255,255,255,0.2) transparent;
        }
        
        /* Estilos del menú */
        .sidebar details { margin-bottom: 5px; border-radius: 8px; overflow: hidden; }
        .sidebar summary { background: transparent; padding: 12px 15px; font-weight: 500; cursor: pointer; border-radius: 8px; transition: var(--transition); list-style: none; display: flex; justify-content: space-between; align-items: center; color: #94a3b8; }
        .sidebar summary::after { content: '\f107'; font-family: "Font Awesome 6 Free"; font-weight: 900; transition: transform 0.3s; }
        .sidebar details[open] > summary::after { transform: rotate(180deg); }
        .sidebar summary:hover, .sidebar details[open] > summary { background: rgba(255,255,255,0.08); color: #fff; }
        .sidebar .menu-links { padding: 5px 0 5px 10px; background: rgba(0,0,0,0.15); border-left: 2px solid rgba(255,255,255,0.1); margin-left: 10px; }
        .sidebar a { display: flex; align-items: center; color: #cbd5e1; text-decoration: none; padding: 10px 12px; margin: 2px 0; border-radius: 6px; transition: var(--transition); font-size: 0.9rem; }
        .sidebar a:hover { background: rgba(255,255,255,0.1); color: #fff; transform: translateX(3px); }
        .sidebar a .icon { width: 25px; margin-right: 10px; text-align: center; font-size: 1.1rem; }

        /* BOTÓN LOGOUT FIJO AL FINAL */
        .sidebar .logout { 
            display: flex; background: #dc2626; color: white; justify-content: center; align-items: center; 
            padding: 12px; margin-top: auto; /* Empujar al final si sobra espacio */
            border-radius: 8px; font-weight: 600; transition: var(--transition); text-decoration: none; 
            box-shadow: 0 4px 6px rgba(220, 38, 38, 0.3);
            flex-shrink: 0; /* No encoger */
        }
        .sidebar .logout:hover { background: #b91c1c; transform: translateY(-2px); }
        
        /* ---------- MAIN ---------- */
        .main { margin-left: 280px; padding: 30px 40px; width: calc(100% - 280px); transition: margin-left 0.3s ease; }

        .topbar { background: rgba(255, 255, 255, 0.95); backdrop-filter: blur(10px); border-radius: 16px; padding: 15px 30px; display: flex; justify-content: flex-end; align-items: center; margin-bottom: 40px; box-shadow: var(--shadow); border: 1px solid rgba(255,255,255,0.5); }
        .user-info { text-align: right; }
        .user-info strong { font-weight: 600; color: var(--primary); font-size: 1rem; }
        .user-info span { font-size: 0.85rem; color: var(--text-muted); }
        
        .section-title { font-size: 1.5rem; font-weight: 700; color: var(--text-main); margin-bottom: 20px; padding-left: 10px; border-left: 5px solid var(--primary); }

        /* ---------- METRICS ---------- */
        .metrics-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(240px, 1fr)); gap: 25px; margin-bottom: 50px; }
        .metric-card { background: var(--card-bg); padding: 25px; border-radius: 16px; box-shadow: var(--shadow); display: flex; align-items: center; gap: 20px; transition: var(--transition); border: 1px solid #e2e8f0; position: relative; overflow: hidden; }
        .metric-card:hover { transform: translateY(-5px); box-shadow: 0 10px 15px -3px rgb(0 0 0 / 0.1); }
        .metric-card::before { content: ''; position: absolute; left: 0; top: 0; bottom: 0; width: 5px; }

        .card-blue::before { background: #3b82f6; } .card-blue .icon-metric { color: #3b82f6; background: #eff6ff; }
        .card-green::before { background: #10b981; } .card-green .icon-metric { color: #10b981; background: #ecfdf5; }
        .card-purple::before { background: #8b5cf6; } .card-purple .icon-metric { color: #8b5cf6; background: #f5f3ff; }
        .card-orange::before { background: #f59e0b; } .card-orange .icon-metric { color: #f59e0b; background: #fffbeb; }

        .icon-metric { width: 60px; height: 60px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 1.8rem; }
        .metric-card .info .value { font-size: 2rem; font-weight: 700; color: var(--text-main); line-height: 1; margin-bottom: 5px; }
        .metric-card .info .title { font-size: 0.9rem; font-weight: 500; color: var(--text-muted); text-transform: uppercase; letter-spacing: 0.5px; }
        
        /* ---------- QUICK ACCESS ---------- */
        .dashboard-content { display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 25px; }
        .card { background: var(--card-bg); padding: 30px 20px; border-radius: 16px; box-shadow: var(--shadow); text-align: center; text-decoration: none; display: flex; flex-direction: column; align-items: center; justify-content: center; transition: var(--transition); border: 1px solid #f1f5f9; height: 100%; }
        .card:hover { transform: translateY(-8px); box-shadow: 0 20px 25px -5px rgb(0 0 0 / 0.1); border-color: var(--primary); }
        .card .icon { font-size: 3rem; margin-bottom: 15px; line-height: 1; filter: grayscale(20%); transition: var(--transition); }
        .card:hover .icon { filter: grayscale(0%); transform: scale(1.1); }
        .card .text { font-size: 0.95rem; font-weight: 600; color: var(--text-main); line-height: 1.4; }

        /* ---------- RESPONSIVE ---------- */
        .toggle-sidebar {
            display: none;
            position: fixed;
            top: 20px;
            left: 20px;
            background: var(--sidebar-bg);
            color: white;
            padding: 10px 14px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 1.2rem;
            z-index: 2000; /* Más alto que todo */
            box-shadow: var(--shadow);
        }
        @media (max-width: 992px) {
            .toggle-sidebar { display: block; }
            .sidebar { transform: translateX(-100%); }
            .sidebar.active { transform: translateX(0); }
            .main { margin-left: 0; width: 100%; padding: 80px 20px 20px 20px; } /* Padding top para no chocar con el botón */
            .topbar { justify-content: center; text-align: center; margin-top: 0; flex-direction: column; gap: 10px; }
            .user-info { text-align: center; }
        }
    </style>
</head>
<body>

<div class="toggle-sidebar" id="sidebarToggle">
    <i class="fas fa-bars"></i>
</div>

<div class="sidebar" id="sidebarMenu">
    <div class="logo-titulo">
        <img src="images/logo.png" alt="Escuela Porongo">
        <h2>Escuela Porongo</h2>
    </div>

    <div class="menu-scroll-container">
        <?php foreach ($menu_agrupado as $titulo_seccion => $items): if (!empty($items)): ?>
            <details>
                <summary><?php echo htmlspecialchars($titulo_seccion); ?></summary>
                <div class="menu-links">
                    <?php foreach ($items as $item): ?>
                        <a href="<?php echo htmlspecialchars($item['url']); ?>">
                            <span class="icon"><?php echo $item['icono']; ?></span>
                            <span><?php echo htmlspecialchars($item['nombre']); ?></span>
                        </a>
                    <?php endforeach; ?>
                </div>
            </details>
        <?php endif; endforeach; ?>
    </div>

    <a href="logout.php" class="logout">
        <span class="icon"><i class="fas fa-sign-out-alt"></i></span>
        <span>Cerrar Sesión</span>
    </a>
</div>

<div class="main" id="mainContent">
    <div class="topbar">
        <div class="user-info">
            <span>Bienvenido/a,</span><br>
            <strong><?php echo htmlspecialchars($_SESSION['nombre'] . " " . $_SESSION['apellido_paterno']. " " . $_SESSION['apellido_materno']); ?></strong>
            <div style="font-size: 0.8rem; color: #64748b; margin-top: 2px;">
                <i class="fas fa-user-tag"></i> <?php echo htmlspecialchars($_SESSION['rol_nombre']); ?>
            </div>
        </div>
    </div>

    <?php if (isAdmin()): ?>
        <section class="metrics-section">
            <h2 class="section-title">Resumen Institucional</h2>
            <div class="metrics-grid">
                <div class="metric-card card-blue">
                    <span class="icon-metric"><i class="fas fa-user-graduate"></i></span>
                    <div class="info"><div class="value"><?= $total_estudiantes ?></div><div class="title">Estudiantes</div></div>
                </div>
                <div class="metric-card card-green">
                    <span class="icon-metric"><i class="fas fa-chalkboard-teacher"></i></span>
                    <div class="info"><div class="value"><?= $total_docentes ?></div><div class="title">Docentes</div></div>
                </div>
                <div class="metric-card card-purple">
                    <span class="icon-metric"><i class="fas fa-hands-helping"></i></span>
                    <div class="info"><div class="value"><?= $total_asistentes ?></div><div class="title">Asistentes</div></div>
                </div>
                <div class="metric-card card-orange">
                    <span class="icon-metric"><i class="fas fa-chalkboard"></i></span>
                    <div class="info"><div class="value"><?= $total_cursos ?></div><div class="title">Cursos Activos</div></div>
                </div>
            </div>
        </section>
    <?php endif; ?>

    <section class="quick-access-section">
        <h2 class="section-title">Accesos Rápidos</h2>
        <div class="dashboard-content">
            <?php foreach ($permisos_usuario as $url => $nombre): ?>
                <a class="card" href="<?php echo htmlspecialchars($url); ?>">
                    <div class="icon"><?php echo $iconos_permisos[$nombre] ?? $icono_default; ?></div>
                    <div class="text"><?php echo htmlspecialchars($nombre); ?></div>
                </a>
            <?php endforeach; ?>
        </div>
    </section>
</div>

<script>
document.addEventListener("DOMContentLoaded", () => {
    const sidebar = document.getElementById('sidebarMenu');
    const toggleBtn = document.getElementById('sidebarToggle');
    const mainContent = document.getElementById('mainContent');

    // Toggle Menú
    toggleBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        sidebar.classList.toggle('active');
        
        const icon = toggleBtn.querySelector('i');
        if (sidebar.classList.contains('active')) {
            icon.classList.remove('fa-bars');
            icon.classList.add('fa-times');
        } else {
            icon.classList.remove('fa-times');
            icon.classList.add('fa-bars');
        }
    });

    // Cerrar al hacer click fuera
    mainContent.addEventListener('click', () => {
        if (sidebar.classList.contains('active')) {
            sidebar.classList.remove('active');
            const icon = toggleBtn.querySelector('i');
            icon.classList.remove('fa-times');
            icon.classList.add('fa-bars');
        }
    });

    // Abrir menú activo
    const currentPath = window.location.pathname.split('/').pop();
    if (currentPath && currentPath !== 'index.php') {
        const links = document.querySelectorAll('.sidebar a');
        links.forEach(link => {
            const linkPath = link.getAttribute('href').split('/').pop();
            if (linkPath === currentPath) {
                link.style.fontWeight = 'bold';
                link.style.color = '#fff';
                link.style.background = 'rgba(255,255,255,0.1)';
                const detailsParent = link.closest('details');
                if (detailsParent) detailsParent.open = true;
            }
        });
    }
});
</script>
</body>
</html>