<?php
session_start();
header('Content-Type: text/html; charset=utf-8');
include '../includes/db.php';
include '../includes/auth.php';
checkAuth();

$conn->set_charset("utf8");

// --- CONFIGURACIÓN ---
$meses_cabecera = ['Agosto', 'Septiembre', 'Octubre', 'Noviembre'];
$META_HORAS_MINUTOS = 27 * 60; // 27 Horas en minutos (1620 min)

// Helper para formato H:M
function minToTime($minutos) {
    if ($minutos == 0) return "-";
    $h = floor($minutos / 60);
    $m = $minutos % 60;
    return sprintf("%d:%02d", $h, $m);
}

// SQL: Obtener datos agrupados
$sql = "
    SELECT 
        u.id, 
        u.rut, 
        u.nombre, 
        u.apellido_paterno, 
        u.apellido_materno,
        r.mes,
        COALESCE(SUM(r.h_realizadas), 0) as real_min
    FROM usuario u
    LEFT JOIN registro_horas_manual r ON u.id = r.usuario_id
    WHERE u.rol_id = 2 AND u.activo = 1
    GROUP BY u.id, r.mes
    ORDER BY u.apellido_paterno ASC
";

// Procesamiento de datos (Pivote)
$data_docentes = [];
$stmt = $conn->prepare($sql);
$stmt->execute();
$result = $stmt->get_result();

while ($row = $result->fetch_assoc()) {
    $id = $row['id'];
    if (!isset($data_docentes[$id])) {
        $data_docentes[$id] = [
            'info' => ['nombre' => $row['apellido_paterno'] . ' ' . $row['apellido_materno'] . ', ' . $row['nombre'], 'rut' => $row['rut']],
            'total_acumulado' => 0
        ];
    }
    if ($row['mes']) {
        $data_docentes[$id]['meses'][$row['mes']] = $row['real_min'];
        $data_docentes[$id]['total_acumulado'] += $row['real_min'];
    }
}

// EXPORTAR
if (isset($_GET['exportar'])) {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=Reporte_Cumplimiento_27H.csv');
    $output = fopen('php://output', 'w');
    fwrite($output, "\xEF\xBB\xBF"); 

    $headers = ['Docente', 'RUT'];
    foreach ($meses_cabecera as $m) $headers[] = $m;
    $headers[] = 'Total Realizado';
    $headers[] = 'Meta';
    $headers[] = 'Diferencia';
    $headers[] = 'Estado';

    fputcsv($output, $headers, ';');

    foreach ($data_docentes as $docente) {
        $fila = [$docente['info']['nombre'], $docente['info']['rut']];
        foreach ($meses_cabecera as $mes) {
            $fila[] = minToTime($docente['meses'][$mes] ?? 0);
        }
        
        $total = $docente['total_acumulado'];
        $diferencia = $total - $META_HORAS_MINUTOS;
        $estado = ($diferencia >= 0) ? 'Cumplido' : 'Pendiente';
        
        $fila[] = minToTime($total);
        $fila[] = "27:00";
        $fila[] = ($diferencia > 0 ? '+' : '') . minToTime(abs($diferencia));
        $fila[] = $estado;

        fputcsv($output, $fila, ';');
    }
    fclose($output);
    exit();
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <link rel="icon" type="image/png" href="../images/logo.png">
    <title>Control de 27 Horas</title>
    <link rel="stylesheet" href="../css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <style>
        .table-responsive-x { overflow-x: auto; max-width: 100%; border: 1px solid #ddd; }
        table { width: 100%; border-collapse: collapse; white-space: nowrap; font-size: 0.9rem; }
        th, td { padding: 8px 10px; border: 1px solid #e0e0e0; text-align: center; }
        thead tr th { background-color: #343a40; color: white; font-weight: bold; }
        
        .sticky-col { position: sticky; left: 0; background-color: #fff; z-index: 2; border-right: 2px solid #ccc; text-align: left; }
        th.sticky-col { background-color: #343a40; color: white; z-index: 3; }

        .badge-ok { background: #e8f5e9; color: #2e7d32; font-weight: bold; padding: 4px 8px; border-radius: 4px; border: 1px solid #c8e6c9; }
        .badge-fail { background: #ffebee; color: #c62828; font-weight: bold; padding: 4px 8px; border-radius: 4px; border: 1px solid #ffcdd2; }
    </style>
</head>
<body>

<header>
    <div class="header-content">
        <img src="../images/INSIGNIA SF.png" alt="Logo">
        <h1>Control de Meta (27 Hrs)</h1>
        <nav>
            <ul>
                <li><a href="../index.php"><i class="fas fa-home"></i> Inicio</a></li>
                <li><a href="registro_horas_admin.php" class="btn btn-secondary">Ingresar Horas</a></li>
            </ul>
        </nav>
    </div>
</header>

<main class="container" style="max-width: 98%;">
    
    <div class="card form-container">
        <div style="display: flex; justify-content: space-between; align-items: center;">
            <h3><i class="fas fa-tasks"></i> Cumplimiento de Horas Realizadas</h3>
            <div style="display: flex; gap: 10px;">
                <a href="?exportar=1" class="btn btn-success"><i class="fas fa-file-excel"></i> Exportar CSV</a>
                <button onclick="window.print()" class="btn btn-secondary"><i class="fas fa-print"></i> Imprimir</button>
            </div>
        </div>
    </div>

    <div class="card table-container" style="margin-top: 15px; padding: 0; overflow: hidden;">
        <div class="table-responsive-x">
            <table>
                <thead>
                    <tr>
                        <th class="sticky-col">Docente</th>
                        <?php foreach ($meses_cabecera as $mes): ?>
                            <th><?= substr($mes,0,3) ?></th>
                        <?php endforeach; ?>
                        <th style="background: #eee; color: #333;">Total</th>
                        <th style="background: #222;">Meta</th>
                        <th style="background: #eee; color: #333;">Diferencia</th>
                        <th>Estado</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($data_docentes as $id => $data): 
                        $total = $data['total_acumulado'];
                        $diferencia = $total - $META_HORAS_MINUTOS;
                        $texto_dif = minToTime(abs($diferencia));
                        
                        // Determinar estado visual
                        if ($diferencia >= 0) {
                            $clase = 'badge-ok';
                            $icono = '<i class="fas fa-check-circle"></i>';
                            $signo = '+';
                            $texto_estado = 'Cumplido';
                        } else {
                            $clase = 'badge-fail';
                            $icono = '<i class="fas fa-exclamation-circle"></i>';
                            $signo = '-';
                            $texto_estado = 'Falta';
                        }
                    ?>
                        <tr>
                            <td class="sticky-col"><strong><?= htmlspecialchars($data['info']['nombre']) ?></strong></td>
                            
                            <?php foreach ($meses_cabecera as $mes): ?>
                                <td><?= minToTime($data['meses'][$mes] ?? 0) ?></td>
                            <?php endforeach; ?>

                            <td style="background: #f9f9f9; font-weight: bold;"><?= minToTime($total) ?></td>
                            <td style="font-weight: bold;">27:00</td>
                            
                            <td style="background: #f9f9f9;">
                                <span style="<?= $diferencia < 0 ? 'color:red;' : 'color:green;' ?>">
                                    <?= $signo . $texto_dif ?>
                                </span>
                            </td>
                            
                            <td>
                                <span class="<?= $clase ?>"><?= $icono ?> <?= $texto_estado ?></span>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</main>
</body>
</html>