<?php
session_start();
header('Content-Type: text/html; charset=utf-8');
include '../includes/db.php';
include '../includes/auth.php';
checkAuth();

// Solo admin
if (!isAdmin()) { /* ... */ }

$conn->set_charset("utf8");

// --- MANEJO DE MENSAJES ---
$message = '';
$msg_type = '';
if (isset($_SESSION['msg'])) {
    $message = $_SESSION['msg'];
    $msg_type = $_SESSION['msg_type'];
    unset($_SESSION['msg'], $_SESSION['msg_type']);
}

// --- 1. ELIMINAR REGISTRO (GET) ---
if (isset($_GET['eliminar_id'])) {
    $id = intval($_GET['eliminar_id']);
    $stmt = $conn->prepare("DELETE FROM registro_horas_manual WHERE id = ?");
    $stmt->bind_param("i", $id);
    if ($stmt->execute()) {
        $_SESSION['msg'] = "Registro eliminado correctamente.";
        $_SESSION['msg_type'] = "success";
    } else {
        $_SESSION['msg'] = "Error al eliminar.";
        $_SESSION['msg_type'] = "error";
    }
    header("Location: gestion_horas_manuales.php");
    exit();
}

// --- 2. ELIMINACIÓN MÚLTIPLE (POST) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['eliminar_masivo'])) {
    if (!empty($_POST['ids_eliminar'])) {
        $ids = array_map('intval', $_POST['ids_eliminar']);
        $ids_string = implode(',', $ids);
        $sql = "DELETE FROM registro_horas_manual WHERE id IN ($ids_string)";
        
        if ($conn->query($sql)) {
            $_SESSION['msg'] = "Se han eliminado " . count($ids) . " registros correctamente.";
            $_SESSION['msg_type'] = "success";
        } else {
            $_SESSION['msg'] = "Error al eliminar registros masivos.";
            $_SESSION['msg_type'] = "error";
        }
    } else {
        $_SESSION['msg'] = "No seleccionaste ningún registro.";
        $_SESSION['msg_type'] = "warning";
    }
    header("Location: gestion_horas_manuales.php");
    exit();
}

// --- 3. EDITAR REGISTRO (POST) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['editar_registro'])) {
    $id = intval($_POST['registro_id']);
    $h_plan = (intval($_POST['plan_h']) * 60) + intval($_POST['plan_m']);
    $h_real = (intval($_POST['real_h']) * 60) + intval($_POST['real_m']);
    $h_noreal = (intval($_POST['noreal_h']) * 60) + intval($_POST['noreal_m']);

    $stmt = $conn->prepare("UPDATE registro_horas_manual SET h_planificadas=?, h_realizadas=?, h_no_realizadas=? WHERE id=?");
    $stmt->bind_param("iiii", $h_plan, $h_real, $h_noreal, $id);
    
    if ($stmt->execute()) {
        $_SESSION['msg'] = "Registro actualizado correctamente.";
        $_SESSION['msg_type'] = "success";
    } else {
        $_SESSION['msg'] = "Error al actualizar.";
        $_SESSION['msg_type'] = "error";
    }
    header("Location: gestion_horas_manuales.php");
    exit();
}

// --- 4. FILTROS Y PAGINACIÓN ---
$filtro_docente = isset($_GET['docente']) ? intval($_GET['docente']) : 0;
$filtro_mes = isset($_GET['mes']) ? $_GET['mes'] : '';

$whereSQL = "WHERE 1=1"; 
$params = [];
$types = "";

if ($filtro_docente > 0) {
    $whereSQL .= " AND r.usuario_id = ?";
    $params[] = $filtro_docente;
    $types .= "i";
}
if ($filtro_mes != '') {
    $whereSQL .= " AND r.mes = ?";
    $params[] = $filtro_mes;
    $types .= "s";
}

// LÓGICA PAGINACIÓN
$registrosPorPagina = 15;
$paginaActual = isset($_GET['pagina']) ? max(1, intval($_GET['pagina'])) : 1;
$offset = ($paginaActual - 1) * $registrosPorPagina;

$sqlCount = "SELECT COUNT(r.id) as total FROM registro_horas_manual r $whereSQL";
$stmtCount = $conn->prepare($sqlCount);
if (!empty($params)) {
    $stmtCount->bind_param($types, ...$params);
}
$stmtCount->execute();
$totalRegistros = $stmtCount->get_result()->fetch_assoc()['total'];
$totalPaginas = ceil($totalRegistros / $registrosPorPagina);
$stmtCount->close();

// CONSULTA FINAL
$sql = "
    SELECT r.*, u.nombre, u.apellido_paterno, u.apellido_materno
    FROM registro_horas_manual r
    JOIN usuario u ON r.usuario_id = u.id
    $whereSQL
    ORDER BY 
        u.apellido_paterno ASC,
        FIELD(r.mes, 'Agosto', 'Septiembre', 'Octubre', 'Noviembre'),
        FIELD(r.dia, 'General', 'Lunes', 'Miércoles', 'Jueves', 'Viernes')
    LIMIT ? OFFSET ?
";

$stmt = $conn->prepare($sql);
$params[] = $registrosPorPagina;
$params[] = $offset;
$types .= "ii";

if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();

$queryString = http_build_query(['docente' => $filtro_docente, 'mes' => $filtro_mes]);
$docentes = $conn->query("SELECT id, nombre, apellido_paterno FROM usuario WHERE rol_id = 2 ORDER BY apellido_paterno");

function formatHM($minutos) {
    $h = floor($minutos / 60);
    $m = $minutos % 60;
    return sprintf("%02d:%02d", $h, $m);
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../images/logo.png">
    <title>Gestión de Horas</title>
    <link rel="stylesheet" href="../css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    
    <style>
        .btn-sm { padding: 5px 10px; font-size: 0.85rem; margin-right: 5px; }
        .td-plan { color: #0d47a1; font-weight: bold; background-color: #e3f2fd; }
        .td-real { color: #1b5e20; font-weight: bold; background-color: #e8f5e9; }
        .td-noreal { color: #b71c1c; font-weight: bold; background-color: #ffebee; }

        /* Estilos Modal y Checkbox */
        .modal { position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1050; display: none; justify-content: center; align-items: center; }
        .modal.show { display: flex; }
        .modal-content { background: #fff; width: 90%; max-width: 500px; border-radius: 8px; padding: 20px; }
        .modal-header { display: flex; justify-content: space-between; margin-bottom: 15px; border-bottom: 1px solid #eee; padding-bottom: 10px; }
        .time-input-group { display: flex; gap: 5px; align-items: center; }
        .time-input-group input { width: 70px; padding: 8px; text-align: center; border: 1px solid #ccc; border-radius: 4px; }
        
        .check-column { width: 40px; text-align: center; }
        .bulk-actions { display: none; margin-left: 15px; }
        .checkbox-custom { transform: scale(1.2); cursor: pointer; }

        /* --- ESTILOS DE PAGINACIÓN NUMÉRICA --- */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin-top: 20px;
            gap: 5px;
        }
        .page-link {
            padding: 8px 12px;
            border: 1px solid #ddd;
            background-color: #fff;
            color: #333;
            text-decoration: none;
            border-radius: 4px;
            transition: 0.2s;
        }
        .page-link:hover {
            background-color: #f0f0f0;
            border-color: #ccc;
        }
        .page-link.active {
            background-color: var(--primary-color);
            color: white;
            border-color: var(--primary-color);
            font-weight: bold;
        }
        .dots {
            padding: 8px;
            color: #777;
        }
    </style>
</head>
<body>

<header>
    <div class="header-content">
        <img src="../images/INSIGNIA SF.png" alt="Logo">
        <h1>Gestión de Horas</h1>
        <nav>
            <ul>
                <li><a href="../index.php"><i class="fas fa-home"></i> Inicio</a></li>
                <li><a href="registro_horas_admin.php" class="btn btn-primary"><i class="fas fa-plus"></i> Nueva Asignación</a></li>
            </ul>
        </nav>
    </div>
</header>

<main class="container">
    <?php if ($message): ?>
        <div class="mensaje <?= ($msg_type=='success')?'exito':'error' ?> show"><?= $message ?></div>
        <script>setTimeout(() => document.querySelector('.mensaje').remove(), 3000);</script>
    <?php endif; ?>

    <div class="card form-container">
        <h2><i class="fas fa-filter"></i> Filtros</h2>
        <form method="GET" style="display: flex; gap: 15px; flex-wrap: wrap; align-items: flex-end;">
            <div style="flex-grow: 1;">
                <label>Docente:</label>
                <select name="docente" class="form-control">
                    <option value="">Todos los docentes</option>
                    <?php while($d = $docentes->fetch_assoc()): ?>
                        <option value="<?= $d['id'] ?>" <?= ($filtro_docente == $d['id']) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($d['apellido_paterno'] . ' ' . $d['nombre']) ?>
                        </option>
                    <?php endwhile; ?>
                </select>
            </div>
            <div style="width: 200px;">
                <label>Mes:</label>
                <select name="mes" class="form-control">
                    <option value="">Todos</option>
                    <option value="Agosto" <?= ($filtro_mes == 'Agosto') ? 'selected' : '' ?>>Agosto</option>
                    <option value="Septiembre" <?= ($filtro_mes == 'Septiembre') ? 'selected' : '' ?>>Septiembre</option>
                    <option value="Octubre" <?= ($filtro_mes == 'Octubre') ? 'selected' : '' ?>>Octubre</option>
                    <option value="Noviembre" <?= ($filtro_mes == 'Noviembre') ? 'selected' : '' ?>>Noviembre</option>
                </select>
            </div>
            <div>
                <button type="submit" class="btn btn-primary">Filtrar</button>
                <a href="gestion_horas_manuales.php" class="btn btn-secondary">Limpiar</a>
            </div>
        </form>
    </div>

    <form method="POST" id="form-masivo" onsubmit="return confirm('¿Estás seguro de eliminar los registros seleccionados?');">
        
        <div class="card table-container" style="margin-top: 20px;">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                <h2><i class="fas fa-list"></i> Registros (Total: <?= $totalRegistros ?>)</h2>
                <button type="submit" name="eliminar_masivo" id="btn-eliminar-masivo" class="btn btn-danger bulk-actions">
                    <i class="fas fa-trash"></i> Eliminar Seleccionados (<span id="count-selected">0</span>)
                </button>
            </div>

            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th class="check-column">
                                <input type="checkbox" id="select-all" class="checkbox-custom" onclick="toggleSelectAll()">
                            </th>
                            <th>Docente</th>
                            <th>Mes</th>
                            <th>Día</th>
                            <th style="text-align: center;">Planificadas</th>
                            <th style="text-align: center;">Realizadas</th>
                            <th style="text-align: center;">No Real.</th>
                            <th style="text-align: center;">Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ($result->num_rows > 0): ?>
                            <?php while($row = $result->fetch_assoc()): ?>
                                <tr>
                                    <td class="check-column">
                                        <input type="checkbox" name="ids_eliminar[]" value="<?= $row['id'] ?>" class="checkbox-item checkbox-custom" onchange="updateBulkButton()">
                                    </td>
                                    <td><?= htmlspecialchars($row['apellido_paterno'] . ' ' . $row['apellido_materno'] . ', ' . $row['nombre']) ?></td>
                                    <td><span style="font-weight: 500; color: #555;"><?= $row['mes'] ?></span></td>
                                    <td><?= (strpos($row['dia'], 'Recup') !== false) ? '<strong style="color:purple">'.str_replace('Recuperación - ', 'Recup. ', $row['dia']).'</strong>' : (($row['dia'] == 'General') ? '<i>Mes Completo</i>' : $row['dia']) ?></td>
                                    <td style="text-align: center;" class="td-plan"><?= formatHM($row['h_planificadas']) ?></td>
                                    <td style="text-align: center;" class="td-real"><?= formatHM($row['h_realizadas']) ?></td>
                                    <td style="text-align: center;" class="td-noreal"><?= formatHM($row['h_no_realizadas']) ?></td>
                                    <td style="text-align: center;">
                                        <button type="button" class="btn btn-warning btn-sm" onclick='editarRegistro(<?= json_encode($row) ?>)'><i class="fas fa-edit"></i></button>
                                        <a href="?eliminar_id=<?= $row['id'] ?>" class="btn btn-danger btn-sm" onclick="return confirm('¿Estás seguro?');"><i class="fas fa-trash"></i></a>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <tr><td colspan="8" style="text-align: center; padding: 20px;">No se encontraron registros.</td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <?php if ($totalPaginas > 1): ?>
                <div class="pagination">
                    <?php if ($paginaActual > 1): ?>
                        <a href="?pagina=<?= ($paginaActual - 1) ?>&<?= $queryString ?>" class="page-link">&laquo; Anterior</a>
                    <?php endif; ?>

                    <?php
                    $rango = 2; // Rango de números alrededor del actual
                    $inicio = max(1, $paginaActual - $rango);
                    $fin = min($totalPaginas, $paginaActual + $rango);

                    if ($inicio > 1) {
                        echo '<a href="?pagina=1&'.$queryString.'" class="page-link">1</a>';
                        if ($inicio > 2) echo '<span class="dots">...</span>';
                    }

                    for ($i = $inicio; $i <= $fin; $i++) {
                        $active = ($i == $paginaActual) ? 'active' : '';
                        echo '<a href="?pagina='.$i.'&'.$queryString.'" class="page-link '.$active.'">'.$i.'</a>';
                    }

                    if ($fin < $totalPaginas) {
                        if ($fin < $totalPaginas - 1) echo '<span class="dots">...</span>';
                        echo '<a href="?pagina='.$totalPaginas.'&'.$queryString.'" class="page-link">'.$totalPaginas.'</a>';
                    }
                    ?>

                    <?php if ($paginaActual < $totalPaginas): ?>
                        <a href="?pagina=<?= ($paginaActual + 1) ?>&<?= $queryString ?>" class="page-link">Siguiente &raquo;</a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>

        </div>
    </form> 

</main>

<div class="modal" id="modalEditar">
    <div class="modal-content">
        <form method="POST">
            <div class="modal-header">
                <h3>Editar Horas (H:M)</h3>
                <button type="button" onclick="cerrarModal()" style="background:none; border:none; font-size:1.5rem; cursor:pointer;">&times;</button>
            </div>
            <div class="modal-body">
                <input type="hidden" name="registro_id" id="edit_id">
                <p id="edit_info" style="margin-bottom: 15px; font-weight: bold; color: var(--primary-color);"></p>

                <div class="input-row">
                    <label style="color:#0d47a1;">Horas Planificadas:</label>
                    <div class="time-input-group">
                        <input type="number" name="plan_h" id="edit_plan_h" min="0"> :
                        <input type="number" name="plan_m" id="edit_plan_m" min="0" max="59">
                    </div>
                </div>
                <div class="input-row">
                    <label style="color:#1b5e20;">Horas Realizadas:</label>
                    <div class="time-input-group">
                        <input type="number" name="real_h" id="edit_real_h" min="0"> :
                        <input type="number" name="real_m" id="edit_real_m" min="0" max="59">
                    </div>
                </div>
                <div class="input-row">
                    <label style="color:#b71c1c;">Horas No Realizadas:</label>
                    <div class="time-input-group">
                        <input type="number" name="noreal_h" id="edit_noreal_h" min="0"> :
                        <input type="number" name="noreal_m" id="edit_noreal_m" min="0" max="59">
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="cerrarModal()">Cancelar</button>
                <button type="submit" name="editar_registro" class="btn btn-primary">Guardar Cambios</button>
            </div>
        </form>
    </div>
</div>

<script>
    function toggleSelectAll() {
        const master = document.getElementById('select-all');
        document.querySelectorAll('.checkbox-item').forEach(cb => cb.checked = master.checked);
        updateBulkButton();
    }

    function updateBulkButton() {
        const count = document.querySelectorAll('.checkbox-item:checked').length;
        const btn = document.getElementById('btn-eliminar-masivo');
        document.getElementById('count-selected').innerText = count;
        btn.style.display = count > 0 ? 'inline-flex' : 'none';
    }

    const modal = document.getElementById('modalEditar');

    function editarRegistro(data) {
        document.getElementById('edit_id').value = data.id;
        document.getElementById('edit_plan_h').value = Math.floor(data.h_planificadas / 60);
        document.getElementById('edit_plan_m').value = data.h_planificadas % 60;
        document.getElementById('edit_real_h').value = Math.floor(data.h_realizadas / 60);
        document.getElementById('edit_real_m').value = data.h_realizadas % 60;
        document.getElementById('edit_noreal_h').value = Math.floor(data.h_no_realizadas / 60);
        document.getElementById('edit_noreal_m').value = data.h_no_realizadas % 60;
        document.getElementById('edit_info').innerText = `${data.nombre} ${data.apellido_paterno} - ${data.mes}`;
        modal.classList.add('show');
    }

    function cerrarModal() { modal.classList.remove('show'); }
    window.onclick = function(event) { if (event.target == modal) cerrarModal(); }
</script>

</body>
</html>