<?php
// 1. Cargar la librería (Asegúrate de haber instalado via Composer antes)
require '../vendor/autoload.php';

// 2. Incluir conexión y autenticación
include '../includes/db.php';
include '../includes/auth.php';
checkAuth();

// Usar los espacios de nombres de la librería
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

// Limpiar cualquier salida previa para evitar corrupción del archivo
if (ob_get_contents()) ob_end_clean();

$conn->set_charset("utf8");

// 3. Consulta SQL
$sql = "SELECT 
            horario.id, 
            horario.fecha, 
            horario.hora_inicio, 
            horario.hora_fin, 
            COALESCE(curso.nombre, 'Sin curso/Reunión') AS curso_nombre, 
            CONCAT(usuario.nombre, ' ', usuario.apellido_paterno, ' ', usuario.apellido_materno) AS usuario_nombre 
        FROM horario 
        LEFT JOIN curso ON horario.curso_id = curso.id 
        JOIN usuario ON horario.usuario_id = usuario.id
        ORDER BY horario.fecha DESC, horario.hora_inicio ASC";

$result = $conn->query($sql);

// 4. Crear el documento Excel
$spreadsheet = new Spreadsheet();
$sheet = $spreadsheet->getActiveSheet();
$sheet->setTitle('Horarios Agendados');

// --- ENCABEZADOS DE LA TABLA ---
$headers = ['ID', 'Fecha', 'Hora Inicio', 'Hora Fin', 'Curso / Actividad', 'Docente / Usuario'];
$col = 'A';

foreach ($headers as $header) {
    $sheet->setCellValue($col . '1', $header);
    $col++;
}

// --- LLENAR DATOS ---
$rowCount = 2; // Empezamos en la fila 2
while ($row = $result->fetch_assoc()) {
    $sheet->setCellValue('A' . $rowCount, $row['id']);
    // Convertir fecha para que Excel la reconozca (Opcional, aquí va como texto)
    $sheet->setCellValue('B' . $rowCount, date('d/m/Y', strtotime($row['fecha'])));
    $sheet->setCellValue('C' . $rowCount, substr($row['hora_inicio'], 0, 5));
    $sheet->setCellValue('D' . $rowCount, substr($row['hora_fin'], 0, 5));
    $sheet->setCellValue('E' . $rowCount, $row['curso_nombre']);
    $sheet->setCellValue('F' . $rowCount, $row['usuario_nombre']);
    
    $rowCount++;
}

// --- ESTILOS VISUALES (Para que se vea bonito) ---

// 1. Estilo para el Encabezado (Fila 1)
$headerStyle = [
    'font' => [
        'bold' => true,
        'color' => ['rgb' => 'FFFFFF'],
    ],
    'fill' => [
        'fillType' => Fill::FILL_SOLID,
        'startColor' => ['rgb' => '4F81BD'], // Azul corporativo
    ],
    'alignment' => [
        'horizontal' => Alignment::HORIZONTAL_CENTER,
        'vertical' => Alignment::VERTICAL_CENTER,
    ],
    'borders' => [
        'allBorders' => [
            'borderStyle' => Border::BORDER_THIN,
            'color' => ['rgb' => '000000'],
        ],
    ],
];
$sheet->getStyle('A1:F1')->applyFromArray($headerStyle);

// 2. Bordes para todo el contenido
$contentStyle = [
    'borders' => [
        'allBorders' => [
            'borderStyle' => Border::BORDER_THIN,
            'color' => ['rgb' => '000000'],
        ],
    ],
    'alignment' => [
        'vertical' => Alignment::VERTICAL_CENTER,
    ],
];
// Aplicar bordes desde A2 hasta la última fila
$sheet->getStyle('A2:F' . ($rowCount - 1))->applyFromArray($contentStyle);

// 3. Centrar columnas específicas (ID, Fecha, Horas)
$sheet->getStyle('A2:D' . ($rowCount - 1))->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

// 4. Auto-ajustar ancho de las columnas
foreach (range('A', 'F') as $columnID) {
    $sheet->getColumnDimension($columnID)->setAutoSize(true);
}

// --- GENERAR Y DESCARGAR ARCHIVO ---
$filename = "horarios_agendados.xlsx";

header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment;filename="' . $filename . '"');
header('Cache-Control: max-age=0');

$writer = new Xlsx($spreadsheet);
$writer->save('php://output');

$conn->close();
exit();
?>