<?php
require __DIR__ . '/../vendor/autoload.php';
// Incluye la biblioteca PhpSpreadsheet
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;

// session_start() se incluye en auth.php
include '../includes/db.php';
include '../includes/auth.php'; 

checkAuth();
if ($_SESSION['rol'] != 'profesor' && !isAdmin()) {
    header("Location: ../index.php");
    exit();
}

$conn->set_charset("utf8mb4");

// AÑADIDO: Manejo de mensajes de sesión (para mostrarlos en HTML)
$message = '';
$message_type = '';
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message'], $_SESSION['message_type']);
}

// Obtener cursos disponibles
$cursos = [];
$sql = "SELECT id, nombre FROM curso WHERE activo = 1 ORDER BY nombre ASC";
$result = $conn->query($sql);
while ($curso = $result->fetch_assoc()) {
    $cursos[] = $curso;
}

// Descargar plantilla Excel
if (isset($_GET['download_template'])) {
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    $headers = ['Nombre', 'Apellido Paterno', 'Apellido Materno', 'RUT', 'Contraseña'];
    foreach ($headers as $index => $header) {
        $sheet->setCellValueByColumnAndRow($index + 1, 1, $header);
    }
    $filename = "plantilla_estudiantes.xlsx";
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header("Content-Disposition: attachment;filename=\"$filename\"");
    header('Cache-Control: max-age=0');
    $writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
    $writer->save('php://output');
    exit();
}

// Procesar la carga del archivo Excel
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['importar']) && isset($_FILES['file'])) {
    $curso_id = intval($_POST['curso_id']);
    $file = $_FILES['file']['tmp_name'];
    $file_type = $_FILES['file']['type'];
    $allowed_types = ['application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'application/vnd.ms-excel'];

    if (!in_array($file_type, $allowed_types)) {
        $_SESSION['message'] = "Error: Solo se permiten archivos Excel (.xls, .xlsx)";
        $_SESSION['message_type'] = 'error';
        header("Location: " . $_SERVER['PHP_SELF']);
        exit();
    }

    try {
        $spreadsheet = IOFactory::load($file);
        $sheet = $spreadsheet->getActiveSheet();
        $rows = $sheet->toArray();

        $expected_headers = ['Nombre', 'Apellido Paterno', 'Apellido Materno', 'RUT', 'Contraseña'];
        $file_headers = array_map('trim', $rows[0]);

        if ($file_headers !== $expected_headers) {
            $_SESSION['message'] = "Error: El formato del archivo no coincide. Las columnas deben ser: " . implode(', ', $expected_headers);
            $_SESSION['message_type'] = 'error';
            header("Location: " . $_SERVER['PHP_SELF']);
            exit();
        }

        array_shift($rows); // Omitir la fila de encabezado

        $stmtInsert = $conn->prepare("
            INSERT INTO usuario (rut, nombre, apellido_paterno, apellido_materno, contraseña, rol_id, curso_id, activo)
            VALUES (?, ?, ?, ?, ?, ?, ?, 1)
            ON DUPLICATE KEY UPDATE
                nombre = VALUES(nombre),
                apellido_paterno = VALUES(apellido_paterno),
                apellido_materno = VALUES(apellido_materno),
                contraseña = VALUES(contraseña),
                rol_id = VALUES(rol_id),
                curso_id = VALUES(curso_id),
                activo = 1
        ");

        if (!$stmtInsert) {
            throw new Exception("Error preparando la consulta: " . $conn->error);
        }

        // Obtener el ID del rol 'estudiante'
        $rol_result = $conn->query("SELECT id FROM roles WHERE nombre = 'estudiante'");
        $rol_id = $rol_result->fetch_assoc()['id'];
        if (!$rol_id) {
             throw new Exception("Error: No se encontró el rol 'estudiante'.");
        }

        foreach ($rows as $row) {
            // Omitir si la fila está vacía (ej. RUT vacío)
            if (empty(trim($row[3]))) continue; 
            
            $nombre = $conn->real_escape_string(trim($row[0]));
            $apellido_paterno = $conn->real_escape_string(trim($row[1]));
            $apellido_materno = $conn->real_escape_string(trim($row[2]));
            $rut = $conn->real_escape_string(trim($row[3]));
            $contrasena = password_hash(trim($row[4]), PASSWORD_BCRYPT);

            $stmtInsert->bind_param("sssssii", $rut, $nombre, $apellido_paterno, $apellido_materno, $contrasena, $rol_id, $curso_id);
            if (!$stmtInsert->execute()) {
                throw new Exception("Error al insertar/actualizar datos: " . $conn->error);
            }
        }

        $stmtInsert->close();
        $_SESSION['message'] = "Estudiantes importados exitosamente al curso seleccionado.";
        $_SESSION['message_type'] = 'exito'; // MODIFICADO: 'exito'
    } catch (Exception $e) {
        $_SESSION['message'] = "Error: " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    }

    $conn->close();
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../images/logo.png">
    <title>Importar Estudiantes</title>

    <link rel="stylesheet" href="../css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    
    <style>
        .form-group input[type="file"] {
            width: 100%;
            padding: 0.65rem; 
            border: 1px solid var(--grey-medium);
            border-radius: var(--border-radius);
            font-size: 0.9rem;
            background-color: #fff;
            transition: var(--transition-speed);
        }
        .form-group input[type="file"]::file-selector-button {
            background-color: var(--grey-dark);
            color: var(--light-color);
            border: none;
            padding: 0.5rem 1rem;
            border-radius: var(--border-radius);
            cursor: pointer;
            margin-right: 1rem;
        }
    </style>
</head>
<body>
    
<header>
    <div class="header-content">
        <img src="../images/INSIGNIA SF.png" alt="Escuela Porongo Insignia">
        <h1>Importar Estudiantes</h1>
        <nav>
            <ul>
                <li><a href="../index.php"><i class="fas fa-home"></i> Inicio</a></li>
                <li><a href="add_student.php"><i class="fas fa-arrow-left"></i> Volver</a></li>
            </ul>
        </nav>
    </div>
</header>
    
<main class="container">

    <?php if ($message): ?>
        <div class="mensaje <?php echo htmlspecialchars($message_type, ENT_QUOTES, 'UTF-8'); ?> show">
            <?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <div class="card" style="text-align: center;">
        <h2><i class="fas fa-download"></i> Paso 1: Descargar Plantilla</h2>
        <p style="margin: 1rem 0;">Descargue la plantilla de Excel para asegurarse de que el formato de los datos sea correcto.</p>
        <a href="?download_template=true" class="btn btn-success">
            <i class="fas fa-file-excel"></i> Descargar Plantilla
        </a>
    </div>

    <div class="card form-container" style="margin-top: 2rem;">
        <h2><i class="fas fa-upload"></i> Paso 2: Importar Estudiantes</h2>
        
        <form method="POST" action="" enctype="multipart/form-data">
            
            <div class="form-group">
                <label for="curso_id"><i class="fas fa-university"></i> Asignar al Curso:</label>
                <select id="curso_id" name="curso_id" required>
                    <option value="">Seleccione un curso...</option>
                    <?php foreach ($cursos as $curso): ?>
                        <option value="<?php echo $curso['id']; ?>"><?php echo htmlspecialchars($curso['nombre']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-group">
                <label for="file"><i class="fas fa-file-excel"></i> Subir archivo Excel:</label>
                <input type="file" id="file" name="file" accept=".xls, .xlsx" required>
            </div>
            
            <div class="form-buttons">
                <button type="submit" name="importar" class="btn btn-primary">
                    <i class="fas fa-check-circle"></i> Importar Estudiantes
                </button>
            </div>
        </form>
    </div>
</main>
    
<script>
    window.onload = function() {
        const message = document.querySelector('.mensaje');
        if (message) {
            message.classList.add('show');
            setTimeout(() => {
                message.classList.remove('show');
            }, 3000); // 3 segundos
        }
    };
</script>
</body>
</html>