<?php
session_start(); // Iniciar la sesión

header('Content-Type: text/html; charset=utf-8'); // Establecer la codificación de caracteres

include '../includes/db.php'; // Incluir el archivo de conexión a la base de datos
include '../includes/auth.php'; // Incluir el archivo de autenticación
checkAuth(); // Verificar si el usuario está autenticado

// Verificar si el usuario tiene el rol de profesor o es administrador
if ($_SESSION['rol'] != 'profesor' && !isAdmin()) {
    header("Location: index.php"); // Redirigir a la página de inicio si no tiene permisos
    exit(); // Terminar la ejecución del script
}

// Establecer la codificación de caracteres para la conexión de la base de datos
$conn->set_charset("utf8mb4");

// AÑADIDO: Manejo de mensajes de sesión
$message = '';
$message_type = '';
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message'], $_SESSION['message_type']);
}

// Función para validar el RUT
function validarRut($rut) {
    $rut = preg_replace('/[^0-9kK]/', '', $rut); // Eliminar puntos y guiones
    if (strlen($rut) < 2) return false;
    $dv = strtoupper(substr($rut, -1));
    $numero = substr($rut, 0, strlen($rut) - 1);
    if (!is_numeric($numero)) return false; // Añadido para seguridad
    $i = 2;
    $suma = 0;
    foreach (array_reverse(str_split($numero)) as $n) {
        $suma += $n * $i;
        $i = ($i == 7) ? 2 : $i + 1;
    }
    $dvr = 11 - ($suma % 11);
    if ($dvr == 11) $dvr = '0';
    if ($dvr == 10) $dvr = 'K';
    return ($dvr == $dv);
}

// Agregar estudiante manualmente
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['crear'])) {
    $rut = $conn->real_escape_string(trim($_POST['rut']));
    $nombre = $conn->real_escape_string(trim($_POST['nombre']));
    $apellido_paterno = $conn->real_escape_string(trim($_POST['apellido_paterno']));
    $apellido_materno = $conn->real_escape_string(trim($_POST['apellido_materno']));
    $curso_id = intval($_POST['curso_id']);
    $contrasena = trim($_POST['contrasena']);

    if (empty($rut) || empty($nombre) || empty($apellido_paterno) || empty($apellido_materno) || empty($contrasena)) {
        $message = "Error: Todos los campos son obligatorios.";
        $message_type = 'error'; // AÑADIDO
    } elseif (!validarRut($rut)) {
        $message = "Error: El RUT ingresado no es válido.";
        $message_type = 'error'; // AÑADIDO
    } else {
        $stmt = $conn->prepare("SELECT id FROM usuario WHERE rut = ?");
        $stmt->bind_param("s", $rut);
        $stmt->execute();
        $stmt->store_result();

        if ($stmt->num_rows > 0) {
            $message = "Error: El RUT ya está registrado.";
            $message_type = 'error'; // AÑADIDO
        } else {
            if (strlen($contrasena) < 8) {
                $message = "Error: La contraseña debe tener al menos 8 caracteres.";
                $message_type = 'error'; // AÑADIDO
            } else {
                $contrasena_hashed = password_hash($contrasena, PASSWORD_BCRYPT);

                $rol_query = $conn->prepare("SELECT id FROM roles WHERE nombre = 'estudiante'");
                $rol_query->execute();
                $rol_result = $rol_query->get_result();
                $rol = $rol_result->fetch_assoc();
                $rol_id = $rol['id'];

                $stmt = $conn->prepare("INSERT INTO usuario (nombre, apellido_paterno, apellido_materno, rut, contraseña, rol_id, curso_id, activo) VALUES (?, ?, ?, ?, ?, ?, ?, 1)");
                $stmt->bind_param("ssssssi", $nombre, $apellido_paterno, $apellido_materno, $rut, $contrasena_hashed, $rol_id, $curso_id);
                
                if ($stmt->execute()) {
                    $message = "Alumno agregado correctamente.";
                    $message_type = 'exito'; // MODIFICADO
                } else {
                    $message = "Error al agregar el alumno: " . $stmt->error;
                    $message_type = 'error'; // AÑADIDO
                }
                $stmt->close();
            }
        }
        $stmt->close(); // Cerrar el statement de verificación de RUT
    }
    
    // MODIFICADO: Usar sesión para mensajes y redirigir
    $_SESSION['message'] = $message;
    $_SESSION['message_type'] = $message_type;
    header("Location: add_student.php");
    exit();
}

// Obtener la lista de cursos
$cursos = $conn->query("SELECT id, nombre FROM curso WHERE activo = 1 ORDER BY nombre ASC");
if ($cursos === FALSE) {
    die("Error al obtener los cursos: " . $conn->error);
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../images/logo.png">
    <title>Agregar Alumnos</title>
    
    <link rel="stylesheet" href="../css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    
    <style>
        .form-group-password {
            position: relative;
        }
        .btn-show-pass {
            position: absolute;
            right: 1px;
            top: 35px; /* Ajustar según la altura del label */
            height: calc(100% - 35px);
            background: none;
            border: none;
            padding: 0 1rem;
            cursor: pointer;
            color: var(--grey-dark);
            border-top-right-radius: var(--border-radius);
            border-bottom-right-radius: var(--border-radius);
        }
        .btn-show-pass:hover {
            color: var(--primary-color);
        }
        #contrasenaHelp {
            font-size: 0.8rem;
            color: var(--grey-dark);
            margin-top: 0.25rem;
        }
    </style>
</head>
<body>
    
<header>
    <div class="header-content">
        <img src="../images/INSIGNIA SF.png" alt="Escuela Porongo Insignia">
        <h1>Agregar Alumnos</h1>
        <nav>
            <ul>
                <li><a href="../index.php"><i class="fas fa-home"></i> Inicio</a></li>
            </ul>
        </nav>
    </div>
</header>
    
<main class="container">

    <?php if ($message): ?>
        <div class="mensaje <?php echo $message_type; ?> show">
            <?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <div class="card form-container">
        <h2><i class="fas fa-user-plus"></i> Agregar Alumno Manualmente</h2>
        
        <form method="POST" action="add_student.php" onsubmit="return validarFormulario()">
            
            <div class="form-grid">
                <div class="form-group">
                    <label for="rut"><i class="fas fa-id-card"></i> RUT (ejemplo: 12345678-9):</label>
                    <input type="text" id="rut" name="rut" placeholder="Ejemplo: 12345678-9" required>
                </div>
                
                <div class="form-group">
                    <label for="nombre"><i class="fas fa-user"></i> Nombre del Alumno:</label>
                    <input type="text" id="nombre" name="nombre" required>
                </div>

                <div class="form-group">
                    <label for="apellido_paterno"><i class="fas fa-user"></i> Apellido Paterno:</label>
                    <input type="text" id="apellido_paterno" name="apellido_paterno" required>
                </div>

                <div class="form-group">
                    <label for="apellido_materno"><i class="fas fa-user"></i> Apellido Materno:</label>
                    <input type="text" id="apellido_materno" name="apellido_materno" required>
                </div>
                
                <div class="form-group form-group-password">
                    <label for="contrasena"><i class="fas fa-lock"></i> Contraseña:</label>
                    <input type="password" id="contrasena" name="contrasena" pattern=".{8,}" title="La contraseña debe tener al menos 8 caracteres" required>
                    <button type="button" id="togglePassword" class="btn-show-pass" title="Mostrar/Ocultar Contraseña">
                        <i class="fas fa-eye"></i>
                    </button>
                    <span id="contrasenaHelp">Mínimo 8 caracteres (ej: abc@2024).</span>
                </div>

                <div class="form-group">
                    <label for="curso_id"><i class="fas fa-university"></i> Curso:</label>
                    <select id="curso_id" name="curso_id" required>
                        <option value="">Seleccione un curso</option>
                        <?php 
                        $cursos->data_seek(0); // Reiniciar puntero
                        while ($curso = $cursos->fetch_assoc()): 
                        ?>
                            <option value="<?php echo htmlspecialchars($curso['id'], ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($curso['nombre'], ENT_QUOTES, 'UTF-8'); ?></option>
                        <?php endwhile; ?>
                    </select>
                </div>
            </div> <div class="form-buttons">
                <button type="submit" name="crear" class="btn btn-primary">
                    <i class="fas fa-plus-circle"></i> Agregar Alumno
                </button>
            </div>
        </form>
    </div> <div class="card" style="margin-top: 2rem; text-align: center;">
        <h2><i class="fas fa-file-excel"></i> Importación Masiva</h2>
        <p style="margin: 1rem 0;">¿Necesitas agregar muchos estudiantes a la vez?</p>
        <a href="import_excel.php" class="btn btn-success">
            <i class="fas fa-upload"></i> Importar desde Excel
        </a>
    </div>

</main>
    
<script>
    // MODIFICADO: Función para el nuevo botón de contraseña
    function togglePasswordVisibility() {
        const passwordField = document.getElementById('contrasena');
        const toggleButtonIcon = document.getElementById('togglePassword').querySelector('i');
        if (passwordField.type === 'password') {
            passwordField.type = 'text';
            toggleButtonIcon.classList.remove('fa-eye');
            toggleButtonIcon.classList.add('fa-eye-slash');
        } else {
            passwordField.type = 'password';
            toggleButtonIcon.classList.remove('fa-eye-slash');
            toggleButtonIcon.classList.add('fa-eye');
        }
    }
    // Asignar el evento al botón
    document.getElementById('togglePassword').addEventListener('click', togglePasswordVisibility);


    // ELIMINADA: showMessage() (manejada por PHP)

    // Función de validación (se mantiene)
    function validarFormulario() {
        const rut = document.getElementById('rut').value.trim();
        const nombre = document.getElementById('nombre').value.trim();
        const apellidoPaterno = document.getElementById('apellido_paterno').value.trim();
        const apellidoMaterno = document.getElementById('apellido_materno').value.trim();
        const contrasena = document.getElementById('contrasena').value.trim();

        if (!rut || !nombre || !apellidoPaterno || !apellidoMaterno || !contrasena) {
            alert('Error: Todos los campos son obligatorios y no pueden estar vacíos o contener solo espacios en blanco.');
            return false;
        }
        
        // Validación de contraseña (opcional, ya que el PHP lo hace)
        if (contrasena.length < 8) {
             alert('Error: La contraseña debe tener al menos 8 caracteres.');
             return false;
        }
        
        return true;
    }
    
    // MODIFICADO: window.onload para manejar el mensaje de sesión
    window.onload = function() {
        const message = document.querySelector('.mensaje');
        if (message) {
            setTimeout(() => {
                message.classList.remove('show');
            }, 3000); // 3 segundos
        }
    };
</script>
</body>
</html>