<?php
session_start(); // Iniciar la sesión
include '../includes/db.php';
include '../includes/auth.php';
checkAuth();

$usuario_id = $_SESSION['usuario_id'];
$conn->set_charset("utf8");

// --- LÓGICA DE ELIMINACIÓN (YA USA SESIONES, ESTÁ CORRECTO) ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['eliminar_equipo'])) {
    $equipo_id = intval($_POST['equipo_id']);
    $conn->begin_transaction();
    try {
        // Eliminar de la tabla `detalles_equipo`
        $stmt_detalles = $conn->prepare("DELETE FROM detalles_equipo WHERE equipo_id = ?");
        $stmt_detalles->bind_param("i", $equipo_id);
        $stmt_detalles->execute();
        $stmt_detalles->close();

        // ... (resto de tus eliminaciones en cascada: mantenimiento, software...)
        $stmt_mantenimiento = $conn->prepare("DELETE FROM mantenimiento_equipo WHERE equipo_id = ?");
        $stmt_mantenimiento->bind_param("i", $equipo_id);
        $stmt_mantenimiento->execute();
        $stmt_mantenimiento->close();

        $stmt_software = $conn->prepare("DELETE FROM software_instalado WHERE equipo_id = ?");
        $stmt_software->bind_param("i", $equipo_id);
        $stmt_software->execute();
        $stmt_software->close();

        // Eliminar de la tabla `equipo`
        $stmt_equipo = $conn->prepare("DELETE FROM equipo WHERE id = ?");
        $stmt_equipo->bind_param("i", $equipo_id);

        if ($stmt_equipo->execute()) {
            $conn->commit();
            $_SESSION['mensaje'] = "Equipo eliminado exitosamente.";
            $_SESSION['mensaje_tipo'] = "exito"; // 'exito' es usado por admin.css
        } else {
            $conn->rollback();
            $_SESSION['mensaje'] = "Error al eliminar el equipo: " . $stmt_equipo->error;
            $_SESSION['mensaje_tipo'] = "error";
        }
        $stmt_equipo->close();
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['mensaje'] = "Error al eliminar el equipo: " . $e->getMessage();
        $_SESSION['mensaje_tipo'] = "error";
    }
    header("Location: crud_equipos.php");
    exit();
}

// --- MANEJO DE MENSAJES DE SESIÓN ---
$message = '';
$message_type = '';
if (isset($_SESSION['mensaje'])) {
    $message = $_SESSION['mensaje'];
    $message_type = $_SESSION['mensaje_tipo'];
    unset($_SESSION['mensaje']);
    unset($_SESSION['mensaje_tipo']);
}

// --- LÓGICA DE BÚSQUEDA Y PAGINACIÓN (CORREGIDA) ---
$limite = 20;
$pagina = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$offset = ($pagina - 1) * $limite;
$search_query_get = ''; // Para los enlaces de paginación

// Definir SQL base y parámetros
$sql_base = "FROM equipo e
             JOIN detalles_equipo d ON e.id = d.equipo_id 
             JOIN estado_equipo es ON e.estado_equipo_id = es.id";
$sql_where = "";
$params = [];
$types = "";

if ($_SERVER['REQUEST_METHOD'] == 'GET' && isset($_GET['search']) && !empty($_GET['search'])) {
    $search = filter_input(INPUT_GET, 'search', FILTER_SANITIZE_SPECIAL_CHARS);
    $search_query_get = htmlspecialchars($search, ENT_QUOTES, 'UTF-8'); // Para la URL
    $search_param = "%$search%";
    
    $sql_where = "WHERE e.nombre_equipo LIKE ? OR d.marca LIKE ? OR d.modelo LIKE ? OR e.ubicacion LIKE ? OR es.nombre_estado LIKE ?";
    $params = [$search_param, $search_param, $search_param, $search_param, $search_param];
    $types = "sssss";
}

// --- CORRECCIÓN 1: Conteo total con filtro ---
$sql_count = "SELECT COUNT(*) as total $sql_base $sql_where";
$stmt_count = $conn->prepare($sql_count);
if (!empty($params)) {
    $stmt_count->bind_param($types, ...$params);
}
$stmt_count->execute();
$total_equipos = $stmt_count->get_result()->fetch_assoc()['total'];
$total_paginas = ceil($total_equipos / $limite);
$stmt_count->close();

// --- CORRECCIÓN 2: Consulta principal con filtro y paginación ---
$sql_query = "SELECT e.id, e.nombre_equipo, d.marca, d.modelo, e.ubicacion, es.nombre_estado 
              $sql_base $sql_where
              ORDER BY e.nombre_equipo ASC
              LIMIT ?, ?";

// Añadir parámetros de paginación
$params[] = $offset;
$params[] = $limite;
$types .= "ii";

$stmt = $conn->prepare($sql_query);
$stmt->bind_param($types, ...$params);

// --- CORRECCIÓN 3: Ejecutar la consulta ---
$stmt->execute();
$equipos = $stmt->get_result();
$stmt->close();
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../images/logo.png">
    <title>Gestionar Equipos</title>
    
    <link rel="stylesheet" href="../css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    
    </head>
<body>

<header>
    <div class="header-content">
        <img src="../images/INSIGNIA SF.png" alt="Escuela Porongo Insignia">
        <h1>Gestionar Equipos</h1>
        <nav>
            <ul>
                <li><a href="../index.php"><i class="fas fa-home"></i> Inicio</a></li>
                <li><a href="equipos.php"><i class="fas fa-plus-circle"></i> Agregar Equipo</a></li>
            </ul>
        </nav>
    </div>
</header>
    
<main class="container">

    <?php if ($message): ?>
        <div class="mensaje <?php echo $message_type; ?> show">
            <?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <div class="card table-container">
        
        <div class="table-header">
            <h2><i class="fas fa-laptop-house"></i> Equipos Registrados</h2>
            <div class="search-and-actions">
                <form class="search-container" method="GET" action="crud_equipos.php">
                    <input type="text" name="search" placeholder="Buscar equipo..." value="<?php echo $search_query_get; ?>">
                    <button type="submit" class="btn btn-primary"><i class="fas fa-search"></i></button>
                </form>
            </div>
        </div>

        <div class="table-responsive">
           <table>
                <thead>
                    <tr>
                        <th>Nombre</th>
                        <th>Marca</th>
                        <th>Modelo</th>
                        <th>Ubicación</th>
                        <th>Estado</th>
                        <th class="actions">Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($equipos->num_rows > 0): ?>
                        <?php while ($equipo = $equipos->fetch_assoc()): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($equipo['nombre_equipo'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($equipo['marca'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($equipo['modelo'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($equipo['ubicacion'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($equipo['nombre_estado'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td class="actions">
                                    
                                    <a href="ver_equipo.php?id=<?php echo $equipo['id']; ?>" class="btn btn-success" title="Ver Detalles">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <a href="editar_equipo.php?id=<?php echo $equipo['id']; ?>" class="btn btn-edit" title="Editar">
                                        <i class="fas fa-pencil-alt"></i>
                                    </a>
                                    <form method="POST" action="crud_equipos.php" style="display:inline;" onsubmit="return confirm('¿Está seguro de que desea eliminar este equipo? Esta acción eliminará todos los detalles, mantenimientos y software asociados.');">
                                        <input type="hidden" name="equipo_id" value="<?php echo $equipo['id']; ?>">
                                        <button type="submit" name="eliminar_equipo" class="btn btn-danger" title="Eliminar">
                                            <i class="fas fa-trash-alt"></i>
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="6" style="text-align: center;">No se encontraron equipos.</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div> <div class="pagination">
            <?php 
            // CORRECCIÓN 3: Añadir búsqueda a la paginación
            $search_url_param = !empty($search_query_get) ? "&search=" . urlencode($search_query_get) : '';
            ?>
            <?php for ($i = 1; $i <= $total_paginas; $i++): ?>
                <a href="?pagina=<?php echo $i; ?><?php echo $search_url_param; ?>" class="<?php echo ($i == $pagina) ? 'current-page' : ''; ?>">
                    <?php echo $i; ?>
                </a>
            <?php endfor; ?>
        </div>
    </div> </main>
    
<script>
window.onload = function() {
    const message = document.querySelector('.mensaje');
    if (message) {
        setTimeout(() => {
            message.classList.remove('show');
        }, 3000);
    }
};
</script>
</body>
</html>