<?php
header('Content-Type: text/html; charset=utf-8');
session_start();
include '../../includes/auth.php';
include '../../includes/db.php';
checkAuth();

// Configuración de errores (ideal para desarrollo)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Variables para mensajes
$message = '';
$message_type = '';

// --- INICIO FUNCIONES AUXILIARES ---

/**
 * Valida un RUT chileno.
 * @param string $rut
 * @return bool
 */
function validarRut($rut) {
    $rut = preg_replace('/[^0-9kK]/', '', $rut);
    if (strlen($rut) < 2) return false;
    
    $dv = strtoupper(substr($rut, -1));
    $numero = substr($rut, 0, strlen($rut) - 1);
    if (!is_numeric($numero)) return false;

    $i = 2;
    $suma = 0;
    foreach (array_reverse(str_split($numero)) as $n) {
        $suma += $n * $i;
        $i = ($i == 7) ? 2 : $i + 1;
    }
    $dvr = 11 - ($suma % 11);
    
    if ($dvr == 11) $dvr = '0';
    if ($dvr == 10) $dvr = 'K';
    
    return ($dvr == $dv);
}

/**
 * Registra una actividad en el log.
 * @param mysqli $conn
 * @param int $usuario_id
 * @param string $accion
 * @param string $descripcion
 */
function registrarActividad($conn, $usuario_id, $accion, $descripcion) {
    $stmt = $conn->prepare("INSERT INTO actividad (usuario_id, accion, descripcion) VALUES (?, ?, ?)");
    $stmt->bind_param("iss", $usuario_id, $accion, $descripcion);
    $stmt->execute();
    $stmt->close();
}

// --- FIN FUNCIONES AUXILIARES ---


// --- INICIO MANEJO DE ACCIONES POST (CRUD) ---

if ($_SERVER['REQUEST_METHOD'] == 'POST') {

    // 1. CREAR O MODIFICAR USUARIO
    if (isset($_POST['crear']) || isset($_POST['modificar'])) {
        $rut = filter_input(INPUT_POST, 'rut', FILTER_SANITIZE_SPECIAL_CHARS);
        $nombre = filter_input(INPUT_POST, 'nombre', FILTER_SANITIZE_SPECIAL_CHARS);
        $apellido_paterno = filter_input(INPUT_POST, 'apellido_paterno', FILTER_SANITIZE_SPECIAL_CHARS);
        $apellido_materno = filter_input(INPUT_POST, 'apellido_materno', FILTER_SANITIZE_SPECIAL_CHARS);
        $contrasena = $_POST['contrasena'];
        $rol_id = filter_input(INPUT_POST, 'rol', FILTER_VALIDATE_INT);
        $curso_id = isset($_POST['curso_id']) && $_POST['curso_id'] !== '' ? filter_input(INPUT_POST, 'curso_id', FILTER_VALIDATE_INT) : NULL;
        $activo = isset($_POST['activo']) ? 1 : 0;

        if (trim($rut) === '' || trim($nombre) === '' || trim($apellido_paterno) === '' || trim($apellido_materno) === '') {
            $message = "Error: Todos los campos (RUT, Nombre, Apellidos) deben contener texto válido.";
            $message_type = 'error';
        } elseif (!validarRut($rut)) {
            $message = "Error: El RUT ingresado no es válido.";
            $message_type = 'error';
        } else {
            // Validar RUT duplicado (solo al crear)
            if (isset($_POST['crear'])) {
                $stmt = $conn->prepare("SELECT id FROM usuario WHERE rut = ?");
                $stmt->bind_param("s", $rut);
                $stmt->execute();
                $stmt->store_result();
                if ($stmt->num_rows > 0) {
                    $message = "Error: El RUT ya está registrado.";
                    $message_type = 'error';
                }
                $stmt->close();
            }

            if (empty($message)) {
                $contrasena_hash = null;
                if (!empty($contrasena)) {
                    if (strlen($contrasena) < 8) {
                        $message = "Error: La contraseña debe tener al menos 8 caracteres.";
                        $message_type = 'error';
                    } else {
                        $contrasena_hash = password_hash($contrasena, PASSWORD_BCRYPT);
                    }
                } elseif (isset($_POST['crear'])) {
                    $message = "Error: La contraseña es requerida para crear un usuario.";
                    $message_type = 'error';
                }

                if (empty($message)) {
                    if (isset($_POST['crear'])) {
                        $stmt = $conn->prepare("INSERT INTO usuario (rut, nombre, apellido_paterno, apellido_materno, contraseña, rol_id, curso_id, activo) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                        $stmt->bind_param("ssssssii", $rut, $nombre, $apellido_paterno, $apellido_materno, $contrasena_hash, $rol_id, $curso_id, $activo);
                        $action = 'creado';
                        $log_action = 'CREAR';
                        $log_desc = "Creó el usuario con RUT $rut";
                    
                    } else if (isset($_POST['modificar'])) {
                        $id = filter_input(INPUT_POST, 'usuario_id', FILTER_VALIDATE_INT);
                        if ($contrasena_hash) {
                            $stmt = $conn->prepare("UPDATE usuario SET rut=?, nombre=?, apellido_paterno=?, apellido_materno=?, contraseña=?, rol_id=?, curso_id=?, activo=? WHERE id=?");
                            $stmt->bind_param("ssssssiii", $rut, $nombre, $apellido_paterno, $apellido_materno, $contrasena_hash, $rol_id, $curso_id, $activo, $id);
                        } else {
                            $stmt = $conn->prepare("UPDATE usuario SET rut=?, nombre=?, apellido_paterno=?, apellido_materno=?, rol_id=?, curso_id=?, activo=? WHERE id=?");
                            $stmt->bind_param("sssssiii", $rut, $nombre, $apellido_paterno, $apellido_materno, $rol_id, $curso_id, $activo, $id);
                        }
                        $action = 'modificado';
                        $log_action = 'MODIFICAR';
                        $log_desc = "Modificó el usuario con ID $id (RUT $rut)";
                    }

                    if ($stmt->execute()) {
                        $message = "Usuario $action exitosamente.";
                        $message_type = 'exito';
                        registrarActividad($conn, $_SESSION['usuario_id'], $log_action, $log_desc);
                    } else {
                        $message = "Error: " . $stmt->error;
                        $message_type = 'error';
                    }
                    $stmt->close();
                }
            }
        }
    }
    
    // 2. ELIMINACIÓN MÚLTIPLE (BORRADO DEFINITIVO)
    elseif (isset($_POST['eliminar_varios'])) {
        $ids = isset($_POST['usuario_ids']) ? $_POST['usuario_ids'] : [];
        if (!empty($ids)) {
            $ids_filtrados = array_filter($ids, fn($id) => filter_var($id, FILTER_VALIDATE_INT));
            if (!empty($ids_filtrados)) {
                $placeholders = implode(',', array_fill(0, count($ids_filtrados), '?'));
                $types = str_repeat('i', count($ids_filtrados));
                
                $stmt = $conn->prepare("DELETE FROM usuario WHERE id IN ($placeholders)");
                $stmt->bind_param($types, ...$ids_filtrados);
                
                if ($stmt->execute()) {
                    $message = "Usuarios eliminados exitosamente.";
                    $message_type = 'exito';
                    registrarActividad($conn, $_SESSION['usuario_id'], 'ELIMINAR', "Eliminó permanentemente usuarios con IDs: " . implode(",", $ids_filtrados));
                } else {
                    $message = "Error al eliminar usuarios: " . $stmt->error;
                    $message_type = 'error';
                }
                $stmt->close();
            }
        } else {
            $message = "No se seleccionaron usuarios para eliminar.";
            $message_type = 'error';
        }
    }

    // 3. INACTIVACIÓN MÚLTIPLE (NUEVA FUNCIONALIDAD AGREGADA)
    elseif (isset($_POST['inactivar_varios'])) {
        $ids = isset($_POST['usuario_ids']) ? $_POST['usuario_ids'] : [];
        if (!empty($ids)) {
            $ids_filtrados = array_filter($ids, fn($id) => filter_var($id, FILTER_VALIDATE_INT));
            if (!empty($ids_filtrados)) {
                $placeholders = implode(',', array_fill(0, count($ids_filtrados), '?'));
                $types = str_repeat('i', count($ids_filtrados));
                
                // Actualizamos el campo activo a 0
                $stmt = $conn->prepare("UPDATE usuario SET activo = 0 WHERE id IN ($placeholders)");
                $stmt->bind_param($types, ...$ids_filtrados);
                
                if ($stmt->execute()) {
                    $message = "Usuarios marcados como inactivos exitosamente.";
                    $message_type = 'exito';
                    registrarActividad($conn, $_SESSION['usuario_id'], 'INACTIVAR_MASIVO', "Inactivó masivamente usuarios con IDs: " . implode(",", $ids_filtrados));
                } else {
                    $message = "Error al inactivar usuarios: " . $stmt->error;
                    $message_type = 'error';
                }
                $stmt->close();
            }
        } else {
            $message = "No se seleccionaron usuarios para inactivar.";
            $message_type = 'error';
        }
    }

    // 4. CAMBIO DE ESTADO A INACTIVO (ELIMINACIÓN LÓGICA INDIVIDUAL)
    elseif (isset($_POST['marcar_inactivo'])) {
        $id = filter_input(INPUT_POST, 'usuario_id', FILTER_VALIDATE_INT);
        if ($id) {
            $stmt = $conn->prepare("UPDATE usuario SET activo = 0 WHERE id = ?");
            $stmt->bind_param("i", $id);
            if ($stmt->execute()) {
                $message = "Usuario marcado como inactivo exitosamente.";
                $message_type = 'exito';
                registrarActividad($conn, $_SESSION['usuario_id'], 'INACTIVAR', "Marcó como inactivo al usuario con ID $id");
            } else {
                $message = "Error al marcar como inactivo: " . $stmt->error;
                $message_type = 'error';
            }
            $stmt->close();
        } else {
            $message = "ID de usuario inválido.";
            $message_type = 'error';
        }
    }
}

// --- FIN MANEJO DE ACCIONES POST ---


// --- INICIO LÓGICA DE CONSULTA Y PAGINACIÓN ---

// 1. Construir cláusula WHERE para búsqueda
$where_clauses = [];
$bind_types = "";
$bind_params = [];
$search_get_param = ''; // Para paginación

if ($_SERVER['REQUEST_METHOD'] == 'GET' && isset($_GET['buscar']) && trim($_GET['search']) !== '') {
    $search_term = "%" . $conn->real_escape_string(trim($_GET['search'])) . "%";
    $search_get_param = htmlspecialchars(trim($_GET['search']), ENT_QUOTES, 'UTF-8');
    
    $where_clauses[] = "(rut LIKE ? OR nombre LIKE ? OR apellido_paterno LIKE ? OR apellido_materno LIKE ? OR curso_id IN (SELECT id FROM curso WHERE nombre LIKE ?) OR rol_id IN (SELECT id FROM roles WHERE nombre LIKE ?))";
    $bind_types = "ssssss";
    $bind_params = [$search_term, $search_term, $search_term, $search_term, $search_term, $search_term];
} else {
    // Por defecto, mostrar solo usuarios activos si no hay búsqueda
    // NOTA: Si deseas ver los inactivos también por defecto, comenta la siguiente línea:
    // $where_clauses[] = "activo = 1"; 
    // Para administración usualmente es mejor ver todos o tener un filtro. Dejaré tu lógica original.
    $where_clauses[] = "activo = 1";
}
$sql_where = !empty($where_clauses) ? "WHERE " . implode(' AND ', $where_clauses) : "";


// 2. Paginación y Conteo
$registros_por_pagina = 20;
$pagina_actual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$offset = ($pagina_actual - 1) * $registros_por_pagina;

// (CORREGIDO) Contar total de usuarios CON filtros
$sql_count = "SELECT COUNT(*) AS total FROM usuario $sql_where";
$stmt_count = $conn->prepare($sql_count);
if (!empty($bind_params)) {
    $stmt_count->bind_param($bind_types, ...$bind_params);
}
$stmt_count->execute();
$total_usuarios = $stmt_count->get_result()->fetch_assoc()['total'];
$total_paginas = ceil($total_usuarios / $registros_por_pagina);
$stmt_count->close();


// 3. Consulta Principal de Usuarios (con paginación)
$sql_query = "SELECT * FROM usuario $sql_where ORDER BY nombre ASC LIMIT ? OFFSET ?";
$stmt_usuarios = $conn->prepare($sql_query);

// Añadir params de paginación
$final_params = array_merge($bind_params, [$registros_por_pagina, $offset]);
$final_types = $bind_types . "ii"; // 'ii' para LIMIT y OFFSET

$stmt_usuarios->bind_param($final_types, ...$final_params);
$stmt_usuarios->execute();
$usuarios = $stmt_usuarios->get_result();
$stmt_usuarios->close();


// 4. Obtener Cursos y Roles (sin duplicados)
$cursos = $conn->query("SELECT id, nombre FROM curso ORDER BY nombre ASC");
$roles = $conn->query("SELECT id, nombre FROM roles ORDER BY nombre ASC");

if ($cursos === FALSE) {
    die("Error al obtener los cursos: " . $conn->error);
}
if ($roles === FALSE) {
    die("Error al obtener los roles: " . $conn->error);
}

// --- FIN LÓGICA DE CONSULTA ---
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../../images/logo.png">
    <title>Administrar Usuarios</title>
    <link rel="stylesheet" href="../../css/admin.css"> 
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
</head>
<body>
    <header>
        <div class="header-content">
            <img src="../../images/INSIGNIA%20SF.png" alt="Logo de la Empresa">
            <h1>Administrar Usuarios</h1>
            <nav>
                <ul>
                    <li><a href="../../index.php"><i class="fas fa-home"></i> Inicio</a></li>
                </ul>
            </nav>
        </div>
    </header>

    <main class="container">
        
        <div class="card form-container">
            <h2><i class="fas fa-user-plus"></i> <span id="form-titulo">Crear Nuevo Usuario</span></h2>
            <form id="formulario-usuario" method="POST" action="admin.php">
                <input type="hidden" name="usuario_id" id="usuario_id">
                
                <div class="form-grid">
                    <div class="form-group">
                        <label for="rut"><i class="fas fa-id-card"></i> RUT:</label>
                        <input type="text" id="rut" name="rut" placeholder="12345678-9" required>
                    </div>
                    <div class="form-group">
                        <label for="nombre"><i class="fas fa-user"></i> Nombre:</label>
                        <input type="text" id="nombre" name="nombre" required>
                    </div>
                    <div class="form-group">
                        <label for="apellido_paterno"><i class="fas fa-user"></i> Apellido Paterno:</label>
                        <input type="text" id="apellido_paterno" name="apellido_paterno" required>
                    </div>
                    <div class="form-group">
                        <label for="apellido_materno"><i class="fas fa-user"></i> Apellido Materno:</label>
                        <input type="text" id="apellido_materno" name="apellido_materno" required>
                    </div>
                    <div class="form-group form-group-password">
                        <label for="contrasena"><i class="fas fa-lock"></i> Contraseña:</label>
                        <input type="password" id="contrasena" name="contrasena" pattern=".{8,}" title="La contraseña debe tener al menos 8 caracteres" required>
                        <button type="button" id="togglePassword" class="btn-show-pass" title="Mostrar/Ocultar Contraseña">
                            <i class="fas fa-eye"></i>
                        </button>
                        <span id="contrasenaHelp">Mínimo 8 caracteres. (Dejar en blanco al editar para no cambiarla)</span>
                    </div>
                    <div class="form-group">
                        <label for="rol"><i class="fas fa-user-tag"></i> Rol:</label>
                        <select id="rol" name="rol" required>
                            <?php $roles->data_seek(0); // Reiniciar puntero ?>
                            <?php while ($rol = $roles->fetch_assoc()): ?>
                                <option value="<?php echo htmlspecialchars($rol['id'], ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($rol['nombre'], ENT_QUOTES, 'UTF-8'); ?></option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="curso_id"><i class="fas fa-chalkboard-teacher"></i> Curso:</label>
                        <select id="curso_id" name="curso_id">
                            <option value="">Ninguno</option>
                            <?php $cursos->data_seek(0); // Reiniciar puntero ?>
                            <?php while ($curso = $cursos->fetch_assoc()): ?>
                                <option value="<?php echo htmlspecialchars($curso['id'], ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($curso['nombre'], ENT_QUOTES, 'UTF-8'); ?></option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    <div class="form-group form-group-checkbox">
                        <label for="activo"><i class="fas fa-check-circle"></i> Activo:</label>
                        <input type="checkbox" id="activo" name="activo" checked>
                    </div>
                </div>
                
                <div class="form-buttons">
                    <button type="submit" name="crear" id="crear" class="btn btn-primary"><i class="fas fa-plus-circle"></i> Crear Usuario</button>
                    <button type="submit" name="modificar" id="modificar" class="btn btn-primary" style="display:none;"><i class="fas fa-save"></i> Guardar Cambios</button>
                    <button type="button" onclick="cancelarEdicion()" class="btn btn-secondary" style="display:none;" id="cancelar"><i class="fas fa-times-circle"></i> Cancelar</button>
                </div>
            </form>
        </div>

        <?php if ($message): ?>
            <div class="mensaje <?php echo $message_type; ?> show">
                <?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?>
            </div>
        <?php endif; ?>

        <div class="card table-container">
            <div class="table-header">
                <h2><i class="fas fa-users"></i> Usuarios Existentes</h2>
                <div class="search-and-actions">
                    <form class="search-container" method="GET" action="admin.php">
                        <input type="text" name="search" placeholder="Buscar usuario..." value="<?php echo $search_get_param; ?>">
                        <button type="submit" name="buscar" class="btn btn-primary"><i class="fas fa-search"></i></button>
                    </form>
                    <form method="post" action="exportar_excel.php" class="export-form">
                        <button type="submit" class="btn btn-success"><i class="fas fa-file-excel"></i> Exportar a Excel</button>
                    </form>
                </div>
            </div>
            
            <form method="POST" action="admin.php">
                <div class="table-responsive">
                    <table>
                        <thead>
                            <tr>
                                <th class="checkbox-column">
                                    <input type="checkbox" onclick="toggleAllCheckboxes(this)">
                                </th>
                                <th>RUT</th>
                                <th>Nombre Completo</th>
                                <th>Rol</th>
                                <th>Curso</th>
                                <th>Estado</th>
                                <th class="actions">Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($usuarios->num_rows > 0): ?>
                                <?php while ($row = $usuarios->fetch_assoc()): ?>
                                    <tr class="<?php echo $row['activo'] ? 'status-active' : 'status-inactive'; ?>">
                                        <td class="checkbox-column">
                                            <input type="checkbox" name="usuario_ids[]" value="<?php echo $row['id']; ?>">
                                        </td>
                                        <td><?php echo htmlspecialchars($row['rut'], ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td><?php echo htmlspecialchars($row['nombre'] . ' ' . $row['apellido_paterno'] . ' ' . $row['apellido_materno'], ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td>
                                            <?php 
                                            $rol_result = $conn->query("SELECT nombre FROM roles WHERE id = " . (int)$row['rol_id']);
                                            echo ($rol_result && $rol_result->num_rows > 0) ? htmlspecialchars($rol_result->fetch_assoc()['nombre'], ENT_QUOTES, 'UTF-8') : 'N/A';
                                            ?>
                                        </td>
                                        <td>
                                            <?php 
                                            if (isset($row['curso_id']) && $row['curso_id']) {
                                                $curso_result = $conn->query("SELECT nombre FROM curso WHERE id = " . (int)$row['curso_id']);
                                                echo ($curso_result && $curso_result->num_rows > 0) ? htmlspecialchars($curso_result->fetch_assoc()['nombre'], ENT_QUOTES, 'UTF-8') : 'N/A';
                                            } else {
                                                echo 'N/A';
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <span class="status-badge <?php echo $row['activo'] ? 'status-active-badge' : 'status-inactive-badge'; ?>">
                                                <?php echo $row['activo'] ? 'Activo' : 'Inactivo'; ?>
                                            </span>
                                        </td>
                                        <td class="actions">
                                            <button type="button" class="btn btn-edit" title="Editar" onclick="editarUsuario('<?php echo $row['id']; ?>', '<?php echo htmlspecialchars($row['rut'], ENT_QUOTES, 'UTF-8'); ?>', '<?php echo htmlspecialchars($row['nombre'], ENT_QUOTES, 'UTF-8'); ?>', '<?php echo htmlspecialchars($row['apellido_paterno'], ENT_QUOTES, 'UTF-8'); ?>', '<?php echo htmlspecialchars($row['apellido_materno'], ENT_QUOTES, 'UTF-8'); ?>', '<?php echo $row['rol_id']; ?>', '<?php echo $row['curso_id']; ?>', '<?php echo $row['activo']; ?>')">
                                                <i class="fas fa-pencil-alt"></i>
                                            </button>
                                            <button type="button" class="btn btn-delete" title="Marcar como Inactivo" onclick="eliminarUsuarioLogico('<?php echo $row['id']; ?>')">
                                                <i class="fas fa-user-slash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="7" style="text-align:center;">No se encontraron usuarios.</td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                <div class="table-footer">
                    <button type="submit" name="inactivar_varios" class="btn" style="background-color: #f0ad4e; color: white; margin-right: 10px;" onclick="return confirm('¿Estás seguro de que quieres MARCAR COMO INACTIVOS a los usuarios seleccionados?');">
                        <i class="fas fa-ban"></i> Inactivar Seleccionados
                    </button>

                    <button type="submit" name="eliminar_varios" class="btn btn-danger" onclick="return confirm('¿Estás seguro de que quieres ELIMINAR PERMANENTEMENTE los usuarios seleccionados?');">
                        <i class="fas fa-trash-alt"></i> Eliminar Seleccionados (Permanente)
                    </button>
                </div>
            </form>

            <?php if ($total_paginas > 1): ?>
                <div class="pagination">
                    <?php for ($i = 1; $i <= $total_paginas; $i++): ?>
                        <a href="?pagina=<?php echo $i; ?>&search=<?php echo $search_get_param; ?><?php echo !empty($search_get_param) ? '&buscar=' : ''; ?>" 
                           class="<?php echo ($i == $pagina_actual) ? 'current-page' : ''; ?>">
                            <?php echo $i; ?>
                        </a>
                    <?php endfor; ?>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <script>
        function togglePasswordVisibility() {
            const passwordField = document.getElementById('contrasena');
            const toggleButtonIcon = document.getElementById('togglePassword').querySelector('i');
            if (passwordField.type === 'password') {
                passwordField.type = 'text';
                toggleButtonIcon.classList.remove('fa-eye');
                toggleButtonIcon.classList.add('fa-eye-slash');
            } else {
                passwordField.type = 'password';
                toggleButtonIcon.classList.remove('fa-eye-slash');
                toggleButtonIcon.classList.add('fa-eye');
            }
        }
        // Asignar el evento al botón
        document.getElementById('togglePassword').addEventListener('click', togglePasswordVisibility);


        function editarUsuario(id, rut, nombre, apellido_paterno, apellido_materno, rol_id, curso_id, activo) {
            document.getElementById('form-titulo').innerHTML = 'Editar Usuario';
            document.getElementById('usuario_id').value = id;
            document.getElementById('rut').value = rut;
            document.getElementById('nombre').value = nombre;
            document.getElementById('apellido_paterno').value = apellido_paterno;
            document.getElementById('apellido_materno').value = apellido_materno;
            document.getElementById('rol').value = rol_id;
            document.getElementById('curso_id').value = curso_id || '';
            document.getElementById('activo').checked = activo === '1';
            
            // Ocultar crear, mostrar modificar/cancelar
            document.getElementById('crear').style.display = 'none';
            document.getElementById('modificar').style.display = 'inline-block';
            document.getElementById('cancelar').style.display = 'inline-block';
            
            // Contraseña no es requerida al editar
            document.getElementById('contrasena').required = false;
            
            // Scroll al formulario
            document.getElementById('formulario-usuario').scrollIntoView({
                behavior: 'smooth',
                block: 'start'
            });
        }

        function cancelarEdicion() {
            document.getElementById('form-titulo').innerHTML = 'Crear Nuevo Usuario';
            document.getElementById('formulario-usuario').reset(); // Limpia el formulario
            
            document.getElementById('usuario_id').value = '';
            document.getElementById('activo').checked = true; // Default a activo

            // Ocultar modificar/cancelar, mostrar crear
            document.getElementById('crear').style.display = 'inline-block';
            document.getElementById('modificar').style.display = 'none';
            document.getElementById('cancelar').style.display = 'none';
            
            // Contraseña vuelve a ser requerida
            document.getElementById('contrasena').required = true;
        }

        /**
         * MODIFICADO: Esta función ahora llama al "borrado lógico" (marcar inactivo)
         * que es más seguro.
         */
        function eliminarUsuarioLogico(id) {
            if (confirm('¿Estás seguro de que quieres marcar este usuario como INACTIVO?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = 'admin.php';
                
                const inputId = document.createElement('input');
                inputId.type = 'hidden';
                inputId.name = 'usuario_id';
                inputId.value = id;
                
                const inputAccion = document.createElement('input');
                inputAccion.type = 'hidden';
                inputAccion.name = 'marcar_inactivo'; // Activa el bloque correcto en PHP
                inputAccion.value = '1';
                
                form.appendChild(inputId);
                form.appendChild(inputAccion);
                document.body.appendChild(form);
                form.submit();
            }
        }

        function toggleAllCheckboxes(source) {
            const checkboxes = document.getElementsByName('usuario_ids[]');
            for (let i = 0; i < checkboxes.length; i++) {
                checkboxes[i].checked = source.checked;
            }
        }

        // Manejo del mensaje de alerta para que desaparezca
        window.onload = function() {
            const message = document.querySelector('.mensaje');
            if (message) {
                setTimeout(() => {
                    message.classList.remove('show');
                }, 3000); // El mensaje desaparece después de 3 segundos
            }
        };
    </script>
</body>
</html>