<?php
session_start(); // Iniciar la sesión
include '../../includes/auth.php';
include '../../includes/db.php';
checkAuth();

$usuario_id = $_SESSION['usuario_id'];
$conn->set_charset("utf8");

$message = ''; // Variable para almacenar el mensaje
$message_type = ''; // Variable para el tipo de mensaje (success o error)

// Obtener mensaje de sesión si existe
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message'], $_SESSION['message_type']);
}

// Eliminar préstamo
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['eliminar_prestamo'])) {
    $prestamo_id = intval($_POST['prestamo_id']);

    try {
        $stmt_eliminar = $conn->prepare("DELETE FROM prestamos WHERE id = ?");
        $stmt_eliminar->bind_param("i", $prestamo_id);
        $stmt_eliminar->execute();
        $stmt_eliminar->close();

        $_SESSION['message'] = "Préstamo eliminado exitosamente.";
        $_SESSION['message_type'] = 'exito'; // Corregido a 'exito'
    } catch (Exception $e) {
        $_SESSION['message'] = "Error al eliminar el préstamo: " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    }

    header("Location: prestamo.php");
    exit();
}

// Editar préstamo
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['editar_prestamo'])) {
    $prestamo_id = intval($_POST['prestamo_id']);
    $fecha_prestamo = $_POST['fecha_prestamo'];
    $fecha_devolucion = $_POST['fecha_devolucion'] ?? null;
    $equipo_id = intval($_POST['equipo_id']);
    $usuario_id = intval($_POST['usuario_id']);
    $estado = $_POST['estado'];

    try {
        $stmt_editar = $conn->prepare("UPDATE prestamos SET fecha_prestamo = ?, fecha_devolucion = ?, equipo_id = ?, usuario_id = ?, estado = ? WHERE id = ?");
        $stmt_editar->bind_param("ssissi", $fecha_prestamo, $fecha_devolucion, $equipo_id, $usuario_id, $estado, $prestamo_id);
        $stmt_editar->execute();
        $stmt_editar->close();

        $_SESSION['message'] = "Préstamo actualizado exitosamente.";
        $_SESSION['message_type'] = 'exito'; // Corregido a 'exito'
    } catch (Exception $e) {
        $_SESSION['message'] = "Error al actualizar el préstamo: " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    }

    header("Location: prestamo.php");
    exit();
}

// Agregar préstamo
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['agregar_prestamo'])) {
    $fecha_prestamo = $_POST['fecha_prestamo'];
    $fecha_devolucion = $_POST['fecha_devolucion'] ?? null;
    $equipo_id = intval($_POST['equipo_id']);
    $usuario_id = intval($_POST['usuario_id']);
    $estado = 'Prestado';

    try {
        $stmt_agregar = $conn->prepare("INSERT INTO prestamos (fecha_prestamo, fecha_devolucion, equipo_id, usuario_id, estado) VALUES (?, ?, ?, ?, ?)");
        $stmt_agregar->bind_param("ssiss", $fecha_prestamo, $fecha_devolucion, $equipo_id, $usuario_id, $estado);
        $stmt_agregar->execute();
        $stmt_agregar->close();

        $_SESSION['message'] = "Préstamo agregado exitosamente.";
        $_SESSION['message_type'] = 'exito'; // Corregido a 'exito'
    } catch (Exception $e) {
        $_SESSION['message'] = "Error al agregar el préstamo: " . $e->getMessage();
        $_SESSION['message_type'] = 'error';
    }

    header("Location: prestamo.php");
    exit();
}

// Configuración de paginación
$limite = 20;
$pagina = isset($_GET['pagina']) ? intval($_GET['pagina']) : 1;
$offset = ($pagina - 1) * $limite;

// Lógica de Búsqueda y Filtro de Estado
$where_clauses = [];
$params = [];
$types = "";
$search_term = '';
$estado_filtro = '';

if (isset($_GET['search']) && !empty($_GET['search'])) {
    $search_term = '%' . $conn->real_escape_string($_GET['search']) . '%';
    $where_clauses[] = "(e.nombre_equipo LIKE ? OR CONCAT(u.nombre, ' ', u.apellido_paterno, ' ', u.apellido_materno) LIKE ?)";
    $params[] = $search_term;
    $params[] = $search_term;
    $types .= "ss";
}

if (isset($_GET['estado_filtro']) && !empty($_GET['estado_filtro'])) {
    $estado_filtro = $conn->real_escape_string($_GET['estado_filtro']);
    $where_clauses[] = "p.estado = ?";
    $params[] = $estado_filtro;
    $types .= "s";
}

$sql_where = "";
if (!empty($where_clauses)) {
    $sql_where = "AND " . implode(" AND ", $where_clauses);
}

// Obtener préstamos con paginación y búsqueda
$sql_base = "FROM prestamos p
             JOIN equipo e ON p.equipo_id = e.id
             JOIN usuario u ON p.usuario_id = u.id
             WHERE u.id NOT IN (SELECT id FROM usuario WHERE rol_id = (SELECT id FROM roles WHERE nombre = 'Estudiante')) $sql_where";

$sql_select = "SELECT p.id, p.fecha_prestamo, p.fecha_devolucion, e.nombre_equipo, CONCAT(u.nombre, ' ', u.apellido_paterno, ' ', u.apellido_materno) AS nombre_completo, p.estado, p.equipo_id, p.usuario_id
               $sql_base
               ORDER BY p.fecha_prestamo DESC
               LIMIT ?, ?";

$stmt_prestamos = $conn->prepare($sql_select);
$params_select = $params;
$params_select[] = $offset;
$params_select[] = $limite;
$types_select = $types . "ii"; 

$stmt_prestamos->bind_param($types_select, ...$params_select);
$stmt_prestamos->execute();
$result_prestamos = $stmt_prestamos->get_result();

// Obtener el total de registros FILTRADOS
$sql_count = "SELECT COUNT(*) as total $sql_base";
$stmt_count = $conn->prepare($sql_count);
if (!empty($types)) {
    $stmt_count->bind_param($types, ...$params);
}
$stmt_count->execute();
$total_resultados = $stmt_count->get_result()->fetch_assoc()['total'];
$total_paginas = ceil($total_resultados / $limite);
$stmt_count->close();

// *** INICIO DE LA LÓGICA CORREGIDA ***
// Manejo de la edición (se define ANTES de la consulta de equipos)
$prestamo_a_editar = null;
if (isset($_GET['edit'])) {
    $prestamo_id = intval($_GET['edit']);
    $stmt_edit = $conn->prepare("SELECT * FROM prestamos WHERE id = ?");
    $stmt_edit->bind_param("i", $prestamo_id);
    $stmt_edit->execute();
    $prestamo_a_editar = $stmt_edit->get_result()->fetch_assoc();
    $stmt_edit->close();
}

// Obtener equipos disponibles
// Esta consulta ahora usa $prestamo_a_editar para incluir el equipo actual
$equipo_actual_id = $prestamo_a_editar ? intval($prestamo_a_editar['equipo_id']) : 0;
$equipos_disponibles = $conn->query("
    SELECT e.id, e.nombre_equipo 
    FROM equipo e
    WHERE e.id = $equipo_actual_id
    OR e.id NOT IN (
        SELECT equipo_id FROM prestamos WHERE estado = 'Prestado'
    )
    ORDER BY e.nombre_equipo ASC
");
// *** FIN DE LA LÓGICA CORREGIDA ***

// Obtener usuarios disponibles
$stmt_usuario = $conn->prepare("SELECT id, nombre, apellido_paterno, apellido_materno FROM usuario WHERE id NOT IN (SELECT id FROM usuario WHERE rol_id = (SELECT id FROM roles WHERE nombre = 'Estudiante')) ORDER BY nombre ASC");
$stmt_usuario->execute();
$result_usuario = $stmt_usuario->get_result();
$stmt_prestamos->close();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../../images/logo.png">
    <title>Préstamo de Equipos</title>
    
    <link rel="stylesheet" href="../../css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
</head>
<body>

<header>
    <div class="header-content">
        <img src="../../images/INSIGNIA%20SF.png" alt="Escuela Porongo Insignia">
        <h1>Gestión de Préstamos de Equipos</h1>
        <nav>
            <ul>
                <li><a href="../../index.php"><i class="fas fa-home"></i> Inicio</a></li>
            </ul>
        </nav>
    </div>
</header>

<main class="container">

    <?php if ($message): ?>
        <div class="mensaje <?php echo $message_type; ?> show">
            <?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <div class="card form-container">
        <h2 id="form-titulo"><i class="fas fa-plus-circle"></i> <?php echo $prestamo_a_editar ? "Editar Préstamo" : "Agregar Préstamo"; ?></h2>
        
        <form id="formulario-prestamo" method="POST" action="prestamo.php">
            <input type="hidden" name="prestamo_id" id="prestamo_id" value="<?php echo $prestamo_a_editar ? htmlspecialchars($prestamo_a_editar['id'], ENT_QUOTES, 'UTF-8') : ''; ?>">
            
            <div class="form-grid">
                <div class="form-group">
                    <label for="fecha_prestamo"><i class="fas fa-calendar-alt"></i> Fecha de Préstamo:</label>
                    <input type="date" id="fecha_prestamo" name="fecha_prestamo" required 
                           value="<?php echo $prestamo_a_editar ? htmlspecialchars($prestamo_a_editar['fecha_prestamo'], ENT_QUOTES, 'UTF-8') : date('Y-m-d'); ?>">
                </div>
                
                <div class="form-group">
                    <label for="fecha_devolucion"><i class="fas fa-calendar-check"></i> Fecha de Devolución:</label>
                    <input type="date" id="fecha_devolucion" name="fecha_devolucion" 
                           value="<?php echo $prestamo_a_editar ? htmlspecialchars($prestamo_a_editar['fecha_devolucion'], ENT_QUOTES, 'UTF-8') : ''; ?>">
                </div>
                
                <div class="form-group">
                    <label for="equipo_id"><i class="fas fa-laptop-house"></i> Seleccionar Equipo:</label>
                    <select name="equipo_id" id="equipo_id" required>
                        <option value="">Seleccione un equipo</option>
                        <?php 
                        // La consulta PHP ahora provee la lista correcta
                        $equipos_disponibles->data_seek(0);
                        while ($equipo = $equipos_disponibles->fetch_assoc()): 
                        ?>
                            <option value="<?php echo $equipo['id']; ?>"
                                <?php echo ($prestamo_a_editar && $prestamo_a_editar['equipo_id'] == $equipo['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($equipo['nombre_equipo'], ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="usuario_id"><i class="fas fa-user"></i> Seleccionar Usuario:</label>
                    <select name="usuario_id" id="usuario_id" required>
                        <option value="">Seleccione un usuario</option>
                        <?php 
                        $result_usuario->data_seek(0); 
                        while ($usuario = $result_usuario->fetch_assoc()): 
                        ?>
                            <option value="<?php echo $usuario['id']; ?>" 
                                <?php echo $prestamo_a_editar && $prestamo_a_editar['usuario_id'] == $usuario['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($usuario['nombre'], ENT_QUOTES, 'UTF-8') . " " . htmlspecialchars($usuario['apellido_paterno'], ENT_QUOTES, 'UTF-8') . " " . htmlspecialchars($usuario['apellido_materno'], ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="estado"><i class="fas fa-toggle-on"></i> Estado:</label>
                    <select name="estado" id="estado" required>
                        <option value="Prestado" <?php echo ($prestamo_a_editar && $prestamo_a_editar['estado'] == 'Prestado') || !$prestamo_a_editar ? 'selected' : ''; ?>>Prestado</option>
                        <option value="Devuelto" <?php echo $prestamo_a_editar && $prestamo_a_editar['estado'] == 'Devuelto' ? 'selected' : ''; ?>>Devuelto</option>
                    </select>
                </div>
            </div> <div class="form-buttons">
                <button type="submit" name="agregar_prestamo" id="crear" class="btn btn-primary" <?php echo $prestamo_a_editar ? 'style="display:none;"' : ''; ?>>
                    <i class="fas fa-plus-circle"></i> Agregar Préstamo
                </button>
                <button type="submit" name="editar_prestamo" id="modificar" class="btn btn-primary" <?php echo !$prestamo_a_editar ? 'style="display:none;"' : ''; ?>>
                    <i class="fas fa-save"></i> Actualizar Préstamo
                </button>
                <button type="button" onclick="cancelarEdicion()" class="btn btn-secondary" id="cancelar" <?php echo !$prestamo_a_editar ? 'style="display:none;"' : ''; ?>>
                    <i class="fas fa-times-circle"></i> Cancelar
                </button>
            </div>
        </form>
    </div> <div class="card table-container">
        
        <div class="table-header">
            <h2><i class="fas fa-search"></i> Buscar Préstamos</h2>
            <div class="search-and-actions">
                <form class="search-container" method="GET" action="prestamo.php">
                    <input type="text" name="search" placeholder="Buscar por equipo o usuario" 
                           value="<?php echo isset($_GET['search']) ? htmlspecialchars($_GET['search'], ENT_QUOTES, 'UTF-8') : ''; ?>">
                    
                    <select name="estado_filtro">
                        <option value="">Todos los estados</option>
                        <option value="Prestado" <?php echo (isset($_GET['estado_filtro']) && $_GET['estado_filtro'] == 'Prestado') ? 'selected' : ''; ?>>Prestado</option>
                        <option value="Devuelto" <?php echo (isset($_GET['estado_filtro']) && $_GET['estado_filtro'] == 'Devuelto') ? 'selected' : ''; ?>>Devuelto</option>
                    </select>
                    
                    <button type="submit" class="btn btn-primary"><i class="fas fa-search"></i></button>
                </form>
                
                <form method="post" action="exportar_prestamos.php" class="export-form">
                    <button type="submit" class="btn btn-success"><i class="fas fa-file-excel"></i> Exportar</button>
                </form>
            </div>
        </div> <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>Fecha Préstamo</th>
                        <th>Fecha Devolución</th>
                        <th>Equipo</th>
                        <th>Usuario</th>
                        <th>Estado</th>
                        <th class="actions">Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($result_prestamos->num_rows > 0): ?>
                        <?php while ($prestamo = $result_prestamos->fetch_assoc()): ?>
                        <tr>
                            <td><?php echo $prestamo['fecha_prestamo']; ?></td>
                            <td><?php echo $prestamo['fecha_devolucion'] ?: 'N/A'; ?></td>
                            <td><?php echo htmlspecialchars($prestamo['nombre_equipo'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($prestamo['nombre_completo'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td>
                                <span class="status-badge <?php echo $prestamo['estado'] == 'Prestado' ? 'status-inactive-badge' : 'status-active-badge'; ?>">
                                    <?php echo htmlspecialchars($prestamo['estado'], ENT_QUOTES, 'UTF-8'); ?>
                                </span>
                            </td>
                            <td class="actions">
                                
                                <a href="prestamo.php?edit=<?php echo $prestamo['id']; ?>#formulario-prestamo" class="btn btn-edit" title="Editar">
                                    <i class="fas fa-pencil-alt"></i>
                                </a>
                                
                                <form method="POST" action="prestamo.php" style="display: inline;">
                                    <input type="hidden" name="prestamo_id" value="<?php echo $prestamo['id']; ?>">
                                    <button type="submit" name="eliminar_prestamo" class="btn btn-delete" title="Eliminar" onclick="return confirm('¿Estás seguro de que deseas eliminar este préstamo?');">
                                        <i class="fas fa-trash-alt"></i>
                                    </button>
                                </form>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="6" style="text-align:center;">No se encontraron préstamos con los filtros actuales.</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div> <div class="pagination">
            <?php
            $search_param = isset($_GET['search']) ? htmlspecialchars($_GET['search'], ENT_QUOTES, 'UTF-8') : '';
            $estado_param = isset($_GET['estado_filtro']) ? htmlspecialchars($_GET['estado_filtro'], ENT_QUOTES, 'UTF-8') : '';
            ?>
            <?php for ($i = 1; $i <= $total_paginas; $i++): ?>
                <a href="?pagina=<?php echo $i; ?>&search=<?php echo $search_param; ?>&estado_filtro=<?php echo $estado_param; ?>"
                   class="<?php echo ($i == $pagina) ? 'current-page' : ''; ?>">
                   <?php echo $i; ?>
                </a>
            <?php endfor; ?>
        </div>
    </div> </main>

<script>
    /**
     * Rellena el formulario superior (invocado por window.onload).
     */
    function editarPrestamo(id, fecha_prestamo, fecha_devolucion, equipo_id, usuario_id, estado) {
        document.getElementById('form-titulo').innerHTML = '<i class="fas fa-edit"></i> Editar Préstamo';
        document.getElementById('prestamo_id').value = id;
        document.getElementById('fecha_prestamo').value = fecha_prestamo;
        document.getElementById('fecha_devolucion').value = fecha_devolucion;
        document.getElementById('equipo_id').value = equipo_id;
        document.getElementById('usuario_id').value = usuario_id;
        document.getElementById('estado').value = estado;

        // Mostrar botones de editar/cancelar y ocultar el de crear
        document.getElementById('crear').style.display = 'none';
        document.getElementById('modificar').style.display = 'inline-block';
        document.getElementById('cancelar').style.display = 'inline-block';
    }

    /**
     * Limpia el formulario y lo restaura al modo "Crear".
     */
    function cancelarEdicion() {
        // Redirige a la página base sin ?edit=
        window.location.href = 'prestamo.php';
    }

    // Manejo de la página al cargar
    window.onload = function() {
        // Cargar datos para edición si la URL tiene ?edit=
        <?php if ($prestamo_a_editar): ?>
            editarPrestamo(
                '<?php echo $prestamo_a_editar['id']; ?>',
                '<?php echo $prestamo_a_editar['fecha_prestamo']; ?>',
                '<?php echo $prestamo_a_editar['fecha_devolucion']; ?>',
                '<?php echo $prestamo_a_editar['equipo_id']; ?>',
                '<?php echo $prestamo_a_editar['usuario_id']; ?>',
                '<?php echo $prestamo_a_editar['estado']; ?>'
            );
        <?php endif; ?>

        // Ocultar mensaje de alerta después de 3 segundos
        const message = document.querySelector('.mensaje');
        if (message) {
            setTimeout(() => {
                message.classList.remove('show');
            }, 3000);
        }
    };
</script>

</body>
</html>