<?php
header('Content-Type: text/html; charset=utf-8');
session_start();
include '../../includes/auth.php';
include '../../includes/db.php';
checkAuth();

// (Buena práctica) Nos aseguramos de que solo los admins entren aquí
if (!isAdmin()) {
    header("Location: ../index.php");
    exit();
}

error_reporting(E_ALL);
ini_set('display_errors', 1);

$message = '';
$message_type = '';

// Manejo de creación de roles
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['crear_rol'])) {
    $nombre_rol = trim(filter_input(INPUT_POST, 'nombre_rol', FILTER_SANITIZE_SPECIAL_CHARS));

    if (empty($nombre_rol)) {
        $message = "Error: El nombre del rol no puede estar vacío.";
        $message_type = 'error';
    } else {
        $stmt = $conn->prepare("SELECT id FROM roles WHERE nombre = ?");
        $stmt->bind_param("s", $nombre_rol);
        $stmt->execute();
        $stmt->store_result();

        if ($stmt->num_rows > 0) {
            $message = "Error: El rol ya existe.";
            $message_type = 'error';
        } else {
            $stmt->close();
            $stmt = $conn->prepare("INSERT INTO roles (nombre) VALUES (?)");
            $stmt->bind_param("s", $nombre_rol);

            if ($stmt->execute()) {
                $message = "Rol creado exitosamente.";
                $message_type = 'success';
            } else {
                $message = "Error: " . $stmt->error;
                $message_type = 'error';
            }
        }
        $stmt->close();
    }
}

// Manejo de edición de roles
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['editar_rol'])) {
    $rol_id = filter_input(INPUT_POST, 'rol_id', FILTER_VALIDATE_INT);
    $nombre_rol = trim(filter_input(INPUT_POST, 'nombre_rol', FILTER_SANITIZE_SPECIAL_CHARS));

    if (empty($nombre_rol)) {
        $message = "Error: El nombre del rol no puede estar vacío.";
        $message_type = 'error';
    } elseif ($rol_id) {
        $stmt = $conn->prepare("UPDATE roles SET nombre = ? WHERE id = ?");
        $stmt->bind_param("si", $nombre_rol, $rol_id);

        if ($stmt->execute()) {
            $message = "Rol actualizado exitosamente.";
            $message_type = 'success';
        } else {
            $message = "Error: " . $stmt->error;
            $message_type = 'error';
        }
        $stmt->close();
    } else {
        $message = "Error: ID de rol inválido.";
        $message_type = 'error';
    }
}

// Manejo de eliminación de roles
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['eliminar_rol'])) {
    $rol_id = filter_input(INPUT_POST, 'rol_id', FILTER_VALIDATE_INT);

    if ($rol_id) {
        // (Opcional) Deberías verificar si algún usuario tiene este rol antes de borrar
        $stmt_check = $conn->prepare("SELECT COUNT(*) as total FROM usuario WHERE rol_id = ?");
        $stmt_check->bind_param("i", $rol_id);
        $stmt_check->execute();
        $total_usuarios = $stmt_check->get_result()->fetch_assoc()['total'];
        $stmt_check->close();

        if ($total_usuarios > 0) {
            $message = "Error: No se puede eliminar el rol porque está asignado a $total_usuarios usuario(s).";
            $message_type = 'error';
        } else {
            $stmt = $conn->prepare("DELETE FROM roles WHERE id = ?");
            $stmt->bind_param("i", $rol_id);

            if ($stmt->execute()) {
                $message = "Rol eliminado exitosamente.";
                $message_type = 'success';
            } else {
                $message = "Error al eliminar el rol: " . $stmt->error;
                $message_type = 'error';
            }
            $stmt->close();
        }
    } else {
        $message = "Error: No se pudo determinar el ID del rol.";
        $message_type = 'error';
    }
}

// Consultar roles existentes
$roles = $conn->query("SELECT id, nombre FROM roles ORDER BY nombre ASC");

if ($roles === FALSE) {
    die("Error al obtener los roles: " . $conn->error);
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../../images/logo.png">
    <title>Gestionar Roles</title>
    <link rel="stylesheet" href="../../css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
</head>
<body>

<header>
    <div class="header-content">
        <img src="../../images/INSIGNIA%20SF.png" alt="Logo de la Empresa">
        <h1>Gestión de Roles</h1>
        <nav>
            <ul>
                <li><a href="../../index.php"><i class="fas fa-home"></i> Inicio</a></li>
            </ul>
        </nav>
    </div>
</header>

<main class="container">

    <div class="card form-container">
        <h2 id="form-titulo"><i class="fas fa-user-tag"></i> Crear Nuevo Rol</h2>
        <form id="formulario-rol" method="POST" action="">
            <input type="hidden" name="rol_id" id="rol_id">
            <div class="form-grid">
                <div class="form-group">
                    <label for="nombre_rol"><i class="fas fa-tag"></i> Nombre del Rol:</label>
                    <input type="text" id="nombre_rol" name="nombre_rol" required>
                </div>
            </div>
            <div class="form-buttons">
                <button type="submit" name="crear_rol" id="crear" class="btn btn-primary"><i class="fas fa-plus-circle"></i> Crear Rol</button>
                <button type="submit" name="editar_rol" id="modificar" class="btn btn-primary" style="display:none;"><i class="fas fa-save"></i> Guardar Cambios</button>
                <button type="button" onclick="cancelarEdicion()" class="btn btn-secondary" style="display:none;" id="cancelar"><i class="fas fa-times-circle"></i> Cancelar</button>
            </div>
        </form>
    </div>

    <?php if ($message): ?>
        <div class="mensaje <?php echo $message_type; ?> show">
            <?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <div class="card table-container">
        <h2><i class="fas fa-list-ol"></i> Roles Existentes</h2>
        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Nombre</th>
                        <th class="actions">Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($roles->num_rows > 0): ?>
                        <?php while ($rol = $roles->fetch_assoc()): ?>
                            <tr>
                                <td><?php echo $rol['id']; ?></td>
                                <td><?php echo htmlspecialchars($rol['nombre'], ENT_QUOTES, 'UTF-8'); ?></td>
                                
                                <td class="actions">
                                    <button type="button" class="btn btn-edit" title="Editar" 
                                            onclick="editarRol('<?php echo $rol['id']; ?>', '<?php echo htmlspecialchars($rol['nombre'], ENT_QUOTES, 'UTF-8'); ?>')">
                                        <i class="fas fa-pencil-alt"></i>
                                    </button>
                                    
                                    <form method="POST" action="" style="display:inline-block;">
                                        <input type="hidden" name="rol_id" value="<?php echo $rol['id']; ?>">
                                        <button type="submit" name="eliminar_rol" class="btn btn-delete" title="Eliminar" 
                                                onclick="return confirm('¿Estás seguro de que deseas eliminar este rol? Esta acción no se puede deshacer.');">
                                            <i class="fas fa-trash-alt"></i>
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="3" style="text-align:center;">No hay roles creados.</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</main>

<script>
    // Mensaje de error o éxito
    window.onload = function() {
        const message = document.querySelector('.mensaje');
        if (message) {
            setTimeout(() => {
                message.classList.remove('show');
            }, 3000); // El mensaje desaparece después de 3 segundos
        }
    };

    /**
     * Rellena el formulario superior con los datos del rol a editar.
     */
    function editarRol(id, nombre) {
        document.getElementById('form-titulo').innerHTML = '<i class="fas fa-user-tag"></i> Editar Rol';
        document.getElementById('rol_id').value = id;
        document.getElementById('nombre_rol').value = nombre;
        
        // Mostrar botones de editar/cancelar y ocultar el de crear
        document.getElementById('crear').style.display = 'none';
        document.getElementById('modificar').style.display = 'inline-block';
        document.getElementById('cancelar').style.display = 'inline-block';

        // Mover la vista al formulario
        document.getElementById('formulario-rol').scrollIntoView({
            behavior: 'smooth',
            block: 'start'
        });
    }

    /**
     * Limpia el formulario y lo restaura al modo "Crear".
     */
    function cancelarEdicion() {
        document.getElementById('form-titulo').innerHTML = '<i class="fas fa-user-tag"></i> Crear Nuevo Rol';
        document.getElementById('formulario-rol').reset(); // Limpia los campos
        document.getElementById('rol_id').value = '';
        
        // Restaurar botones
        document.getElementById('crear').style.display = 'inline-block';
        document.getElementById('modificar').style.display = 'none';
        document.getElementById('cancelar').style.display = 'none';
    }
</script>

</body>
</html>